' -----------------------------------------------------------------------------
'                     S C R I P T   S P E C I F I C A T I O N
'                    Copyright 2012-2015 Motorola Solutions, Inc.
'                              All Rights Reserved.
'                    Motorola Solutions Confidential Restricted
' -----------------------------------------------------------------------------
'
' NAME
'       CommonFunctions.vbs
'
' DESCRIPTION
'       Common functions and subroutines can be used in all sa scripts
'
' MODIFICATION HISTORY
' Date       CR              Change
' -----------------------------------------------------------------------------
' 11 Oct 2012  CCMPD01711499   Initial creation
' 31 Oct 2012  CCMPD01711499   FTR rework
' 08 Nov 2012  CCMPD01716336   added function for manipulating key registry
' 21 Nov 2012  CCMPD01721207   adding function for checking key registry integrity.
' 29 Nov 2012  CCMPD01721208   Added functions: IsTaskScheduled, DeleteTask
'                              and IsRunningByname
'                              Changed function: StopByName
' 14 Dec 2012  CCMPD01721208   FTR rework
' 14 Dec 2012  CCMPD01721208   1. Removed UninstallApp and GetAppUninstallRegVal
'                                 functions
'                              2. GetAppGUID and IsAppInstalled updates, caused
'                                 by the fact that supp CD have to Java products:
'                                 Java and Java6
' 19 Dec 2012  CCMPD01728163   Fixed error handling in AppVersionComp,
'                              minor changes in other functions
' 02 Jan 2013  CCMPD01728163   FTR rework
' 03 Jan 2013  CCMPD01728186   Remove IACONFIG
' 04 Jan 2013  CCMPD01731447   1. Updates needed to handle Remote Desktop
'                                 Updates product
'                              2. UpgradeTime and PreviousVersion registers are
'                                 now created for thirdparty apps no first
'                                 installation - consistency with Motorola apps
' 18 Jan 2013  CCMPD01734569   Fix IsRemoteDesktopUpdatesInstalled
' 12 Feb 2013  CCMPD01742007   Update CommonFunction.vbs comment to indicate what are impact to GNMO
' 15 Feb 2013  CCMPD01744395   Adjust AppVersionComp (IsValidMSIVersion) to handle both
'                              AA.BB.CC and AA.BB.CC.DD version formats
' 19 Feb 2013  CCMPD01745484   3rd party folder name needs to be updated per sys-033 FTR
' 20 Feb 2013  CCMPD01745484   FTR rework
' 14 Mar 2013  CCMPD01754666   Add seconds to the timestamp
' 23 May 2013  CCMPD01776986   Move PostUpgrade actions to PostInstall
' 10 Jun 2013  CCMPD01776988   Need to handle newer than expected case for third party apps
' 17 Jun 2013  CCMPD01776988   FTR rework
' 19 Jun 2013  CCMPD01785822   Common Agent installation fix
' 19 Aug 2013  CCMPD01802955   Added timeout to StopByName function
' 20 Aug 2013  CCMPD01801690   Stopping scripts for StopByName
' 04 Sep 2013  CCMPD01811323   Added function ConfigJavaDeploymentProperties, which
'                              configures java system deployment properties
' 13 Sep 2013  CCMPD01814607   Java update failed due to deployment.properties file
'                              read-only attribute being set
' 14 Oct 2013  CCMPD01825178   Changes to gracefully shutdown Java Control Panel
' 21 Oct 2013  CCMPD01825178   CloseJavaCPL: fail of soft killing does not fail
'                              whole installation
' 04 Dec 2013  CCMPD01842442   Add OS version check for windows 8 and 8.1
' 29 Nov 2013  CCMPD01840344   Set deployment.expiration.check.enabled to false
'                                 at ConfigJavaDeploymentProperties
' 16 Dec 2013  CCMPD01844578   System upgrade tools
' 24 Dec 2013  CCMPD01847696   Added functions to export data to retention datastore
' 13 Jan 2014  CCMPD01844583   Functions to export registry value
' 25 Feb 2014  CCMPD01866055   Fixed GetAppProgramFiles function'
' 05 May 2014  CCMPD01891253   Not raise error when found undefined OS version
' 31 Jul 2014  CCMPD01917965   Increased duration for PostInstallConfig beacuse
'                              of 600 seconds timeout for javacpl
' 07 Jul 2014  CCMPD01907806   (and others) create java 8 x86, x64 prototype,
'                              rename java 7 as java 7 x86
' 31 Jul 2014  CCMPD01916764   Rename Java 7 x86 to Java 7 Static x86
' 06 Aug 2014  CCMPD01916765   Added SetJavaDeploymentConfigProperty and modified
'                              SetJavaDeploymentProperty
' 09 Oct 2014  CCMPD01916765   Left only creation on deployment.properties in ConfigJavaDeploymentProperties
' 24 Oct 2014  CCMPD01916765   deployment.* in separate locations
' 03 Nov 2014  CCMPD01916765   move copying of deployment.properties files to proper postInstallConfig.vbs
' 06 Nov 2014  CCMPD01916781   Added new version of GetAppBitInfo from fr15663_b-162647
' 01 Dec 2014  B-167882        IE11 - Installation script - postinstall & postupgrade
' 07 Jan 2015  b172792         Exclude IE11 from GUID check
' 18 Feb 2015  B-179526        A716 Motorola OpenJDK terminate script kills all java*.exe processes
' 04 Mar 2015  B-172818        Added Windows Management Framework changes
' 09 Mar 2015  B-152600        IA - [3/6] - [SR CCMPD01969425]
'                              Update version of Java to latest available 1.8 (and 1.7) -final tunning and tests
' 13 Mar 2015  ccmpd01976544   Fix jp2launcher termination.
' 19 Mar 2015  CCMPD01977602   Adding 'keytool.exe' to termination array.
' 01 Jun 2015  CCMPD01998006   Added function checkJnlpAssociation
' 15 Jun 2015  B-202214        Slight refactor of checkJnlpAssociation (now called addRegistryEntries) and implementing Java cache split for jre7
' 30 Jun 2015  CCMPD02006635   Add English - United States locale constant
' 02 Jul 2015  CCMPD02006635   Review rework
' -----------------------------------------------------------------------------
'
'IMPORTANT: global variable logFileName needs to be defined to use all subs and functions from this script
'

Option Explicit

'define constants for all SA scripts
'Constants below are used by NM teams in their scripts. If you changed them them please notify NM Platform team.
'For more details go to CPT vobs : /vobs/cpt/doc/icd/Supplemental\ CD\ framework.docx
'-------------------------------DO NOT MODIFY-----------------------------------------
const SUCCESS = 0
const FAIL = 1
const WARNING = 2
const ERROR_1638 = 1638

const INFO_C    = "Information"
const ERROR_C   = "Error"
const WARNING_C = "Warning"
'--------------------------------END OF SECTION---------------------------------------

const logFilePath = "c:\windows\debug\"

const UNKNOWN_ERROR   = 50000
const OS_BIT_ERROR   = 50001
const APP_GUID_ERROR = 50002
const OS_TYPE_ERROR  = 50003
const INVALID_FUN_PARAM_ERROR = 50004

const OS_BIT = 32
const UNKNOWN_PARAMETER_ERROR = 438

const CURRENTVERSION  = "Version"
const INITIALVERSION  = "InitialVersion"
const PREVIOUSVERSION = "PreviousVersion"
const INITIALTIME     = "InitialInstallTime"
const UPGRADETIME     = "UpgradeTime"
const GUID            = "GUID"
const SACONFIGVER     = "SaVersion"

const HKEY_CLASSES_ROOT   = &H80000000
const HKEY_CURRENT_USER   = &H80000001
const HKEY_LOCAL_MACHINE  = &H80000002
const HKEY_USERS          = &H80000003
const HKEY_CURRENT_CONFIG = &H80000005
const HKEY_DYN_DATA       = &H80000006

const REG_SOFTWARE = "Software"
const REG_SOFTWARE_WOW6432NODE = "Software\WOW6432Node"

const FOLDER_PROGRAM_FILES     = "c:\Program Files"
const FOLDER_PROGRAM_FILES_X86 = "c:\Program Files (x86)"

const PROCESS = 170580
const SERVICE = 170581
const TASK    = 170582
const SCRIPT  = 170583

' Timeout in miliseconds
const DEFAULT_STOP_TIMEOUT      = 20000
const JAVA_DEFAULT_STOP_TIMEOUT = 40000

const CSI_UNKNOWN_ARCH   = 0
const CSI_UNKNOWN_TARGET = 99999
const CSI_OS_BIT_ERROR   = 50001

'OpenTextFile function's file open modes
const FOR_READING   = 1
const FOR_WRITING   = 2
const FOR_APPENDING = 8

'OpenTextFile function's file formats
const TRISTATE_USE_DEFAULT = -2
const TRISTATE_TRUE        = -1
const TRISTATE_FALSE       = 0

' File Attribute Values
const READ_ONLY = 1
const HIDDEN    = 2

' Windows applications
const TASKKILL = "C:\Windows\System32\taskkill.exe"

' Partition used for data retention during major upgrade
const UPGRADE_DATA_STORE_DRIVE = "D:"
const UPGRADE_DATA_STORE_FOLDER = "D:\Upgrade Data Store"
const UPGRADE_DATA_STORE_FILE = "metadata.txt"
const UPGRADE_DATA_STORE_SEPARATOR = ":"

' Prepare for upgrade function exit codes
const PREUPGRADE_SUCCESS = 0
const PREUPGRADE_FAIL    = 1
const PREUPGRADE_SKIP    = 2

' Drive types
const DRIVE_UNKNOWN   = 0
const DRIVE_REMOVABLE = 1
const DRIVE_FIXED     = 2
const DRIVE_NETWORK   = 3
const DRIVE_CDROM     = 4
const DRIVE_RAMDISK   = 5

' File extenstions
const REG_FILE_EXTENSION = "reg"

' Locale IDs (LCID Dec)
' Reference:
' - https://msdn.microsoft.com/en-us/goglobal/bb964664.aspx - Locale IDs Assigned by Microsoft
const LCID_EN_US = 1033

' This is for SYS-033 Product Folder name compliance.
' Creates mapping between SYS-033 naming and real names.
Dim Applications
Set Applications = CreateObject("Scripting.Dictionary")
'----------------------MODIFY ONLY ON SYS-033 CHANGE---------------------------
Applications("Motorola ASTRO 7-Zip") = "7-Zip"
Applications("Motorola ASTRO Adobe Reader") = "Adobe Reader"
Applications("Motorola ASTRO Java 7 Static") = "Java7"
Applications("Motorola ASTRO Java Family x64") = "Java_x64"
Applications("Motorola ASTRO Java Family") = "Java"
'----------------------MODIFY ONLY ON SYS-033 CHANGE---------------------------

' Detection of more than one instance of these applications in PreCheck,
' is not treated as an error.
Dim multipleInstancesApps
Set multipleInstancesApps = CreateObject("Scripting.Dictionary")
multipleInstancesApps("Java") = ""
multipleInstancesApps("Java7") = ""
multipleInstancesApps("Java_x64") = ""
multipleInstancesApps("Microsoft .NET Framework") = ""

Dim systemBitsInfo, processBitsInfo
systemBitsInfo = Null
processBitsInfo = Null

const OS_CAPTION = 1
const OS_VERSION = 2

Dim arrMotorolaJavas(24)
arrMotorolaJavas(0) = "C:\\Program Files (x86)\\Java\\jre8\\bin\\javaws.exe"
arrMotorolaJavas(1) = "C:\\Program Files (x86)\\Java\\jre8\\bin\\javaw.exe"
arrMotorolaJavas(2) = "C:\\Program Files (x86)\\Java\\jre8\\bin\\java.exe"
arrMotorolaJavas(3) = "C:\\Program Files (x86)\\Java\\jre8\\bin\\jp2launcher.exe"
arrMotorolaJavas(4) = "C:\\Program Files (x86)\\Java\\jre8\\bin\\keytool.exe"
arrMotorolaJavas(5) = "C:\\Program Files\\Java\\jre8\\bin\\javaws.exe"
arrMotorolaJavas(6) = "C:\\Program Files\\Java\\jre8\\bin\\javaw.exe"
arrMotorolaJavas(7) = "C:\\Program Files\\Java\\jre8\\bin\\java.exe"
arrMotorolaJavas(8) = "C:\\Program Files\\Java\\jre8\\bin\\keytool.exe"
arrMotorolaJavas(9) = "C:\\Program Files\\Java\\jre8\\bin\\jp2launcher.exe"
arrMotorolaJavas(10) = "C:\\Program Files (x86)\\Java\\jre7\\bin\\javaws.exe"
arrMotorolaJavas(11) = "C:\\Program Files (x86)\\Java\\jre7\\bin\\javaw.exe"
arrMotorolaJavas(12) = "C:\\Program Files (x86)\\Java\\jre7\\bin\\java.exe"
arrMotorolaJavas(13) = "C:\\Program Files (x86)\\Java\\jre7\\bin\\jp2launcher.exe"
arrMotorolaJavas(14) = "C:\\Program Files (x86)\\Java\\jre7\\bin\\keytool.exe"
arrMotorolaJavas(15) = "C:\\Program Files (x86)\\Java\\jre1.6.0_35\\bin\\javaws.exe"
arrMotorolaJavas(16) = "C:\\Program Files (x86)\\Java\\jre1.6.0_35\\bin\\javaw.exe"
arrMotorolaJavas(17) = "C:\\Program Files (x86)\\Java\\jre1.6.0_35\\bin\\java.exe"
arrMotorolaJavas(18) = "C:\\Program Files (x86)\\Java\\jre1.6.0_35\\bin\\jp2launcher.exe"
arrMotorolaJavas(19) = "C:\\Program Files (x86)\\Java\\jre1.6.0_35\\bin\\keytool.exe"
arrMotorolaJavas(20) = "C:\\Program Files (x86)\\Java\\jre1.7.0_79\\bin\\java.exe"
arrMotorolaJavas(21) = "C:\\Program Files (x86)\\Java\\jre1.7.0_79\\bin\\javaw.exe"
arrMotorolaJavas(22) = "C:\\Program Files (x86)\\Java\\jre1.7.0_79\\bin\\javaws.exe"
arrMotorolaJavas(23) = "C:\\Program Files (x86)\\Java\\jre1.7.0_79\\bin\\jp2launcher.exe"
arrMotorolaJavas(24) = "C:\\Program Files (x86)\\Java\\jre1.7.0_79\\bin\\keytool.exe"


'Functions and subs defined in the section below are used by NM teams in their scripts. If you changed them please notify NM Platform team.
'For more details go to CPT vobs : /vobs/cpt/doc/icd/Supplemental\ CD\ framework.docx

'---------------------------------------------DO NOT MODIFY-----------------------------------------------------------------------------------

Sub WriteToLogFile (ByVal msgStr, ByVal msgType)
    ' Writes message to given log file.
    ' We do not need to check if this throw any error if our function already failed and we just loggin error.
    ' It means that in case of logging INFO_C we should always check for errors which this function can throw
    '
    ' [msgStr]      In - message to log.
    ' [msgType]     In - log message type.
    '
    ' Remark
    ' It uses global variable logFileName
    ' it uses WScript.ScriptName to determine scripts calling this sub

    Dim oFSO, oTextString
    Dim sErrorMsg, sTimeStr

    On Error Resume Next

    sTimeStr = CurrentTimeStr

    Set oFSO = CreateObject("Scripting.FileSystemObject")

    If (Err.Number <> 0) Then
        WScript.echo ERROR_C & ":" & sTimeStr & ":" & WScript.ScriptName & ":" & Err.Description
        Exit Sub
    End If

    Set oTextString = oFSO.openTextFile(logFileName, FOR_APPENDING, True)

    If (Err.Number <> 0) Then
        WScript.echo ERROR_C & ":" & sTimeStr & ":" & WScript.ScriptName & ":" & Err.Description
        Set oFSO = Nothing
        Exit Sub
    End If

    oTextString.write msgType & ":" & sTimeStr & ":" & WScript.ScriptName & ":" & msgStr & vbCrLf
    oTextString.close

    Set oFSO = Nothing
    Set oTextString = Nothing

End Sub


Sub WriteToEventLog (ByVal msgStr, ByVal msgType)
    ' Writes message to system log (event log).
    ' We do not need to check if this throw any error if our function already failed and we just loggin error.
    ' It means that in case of logging INFO_C we should always check for errors which this function can throw
    '
    ' [msgStr]      In - message to log.
    ' [msgType]     In - log message type.
    '
    ' Remark
    ' it uses WScript.ScriptName to determine scripts calling this sub

    'code to log to event log
    const EVENT_SUCCESS       = 0
    const EVENT_ERROR         = 1
    const EVENT_WARNING       = 2
    const EVENT_INFORMATION   = 4
    const EVENT_AUDIT_SUCCESS = 8
    const EVENT_AUDIT_FAILURE = 16

    Dim oWshShell, sMsg
    Dim sTimeStr

    sMsg = WScript.ScriptName & ":" & msgStr

    sTimeStr = CurrentTimeStr

    On Error Resume Next

    Set oWshShell = WScript.CreateObject("WScript.Shell")

    If (Err.Number <> 0) Then
        WScript.echo ERROR_C & ":" & sTimeStr & ":" & WScript.ScriptName & ":" & Err.Description
        Exit Sub
    End If

    If (msgType = INFO_C) Then
        oWshShell.LogEvent EVENT_INFORMATION, sMsg
    ElseIf (msgType = ERROR_C) Then
        oWshShell.LogEvent EVENT_ERROR, sMsg
    ElseIf (msgType = WARNING_C) Then
        oWshShell.LogEvent EVENT_WARNING, sMsg
    Else
        oWshShell.LogEvent EVENT_INFORMATION, sMsg
    End If

    Set oWshShell = Nothing

    If (Err.Number <> 0) Then
        WScript.echo ERROR_C & ":" & sTimeStr & ":" & WScript.ScriptName & ":" & Err.Description
        Exit Sub
    End If

End Sub

'--------------------------------------------------END OF SECTION------------------------------------------------------------------------

Function CurrentTimeStr
    ' Returns time string as: Day YYYY-MM-DD HH:MM:SS.
    '
    ' Return value (sample):
    ' Mon 2012-11-05 11:38:14
    Dim myDate, mth, dy
    myDate = Now
    mth = Month(myDate)
    dy = Day(myDate)

    ' Format month and day
    If mth < 10 Then
        mth = "0" & mth
    End If

    If dy < 10 Then
        dy = "0" & dy
    End If

    ' return time string
    CurrentTimeStr = WeekdayName(Weekday(myDate, vbSunday), True, vbSunday) & " " & Year(myDate) & "-" & mth & "-" & dy & " " & FormatDateTime(myDate, vbLongTime)

End Function

Function CurrentTimeGeneralDateStr
    ' Returns time string as: MM/DD/YYYY HH:MM:SS AM/PM
    '
    ' Return value (sample):
    ' 5/11/2012 11:38:13 AM
    Dim myDate
    myDate = Now
    CurrentTimeGeneralDateStr = FormatDateTime(myDate, vbGeneralDate)

End Function


Sub LogError(ByVal sErrMsg)
    ' log an error to log file and event log.
    On Error Resume Next

    WScript.echo sErrMsg

    ' need to write the sErrMsg to a logFile, SA does not have console to view the mssg.
    Call WriteToLogFile(sErrMsg, ERROR_C)

    ' write to event log
    Call WriteToEventLog(sErrMsg, ERROR_C)

End Sub

Sub LogWarning(ByVal sErrMsg)
    ' log an error to log file and event log.
    On Error Resume Next

    WScript.echo sErrMsg

    ' need to write the sErrMsg to a logFile, SA does not have console to view the mssg.
    Call WriteToLogFile(sErrMsg, WARNING_C)

    ' write to event log
    Call WriteToEventLog(sErrMsg, WARNING_C)

End Sub

Function GetAppVersion (ByVal appName)
    ' Gets application version that is displayed in Control Panel.
    '
    ' [appName] In - application name.
    '
    ' Return value:
    '  Empty               - if application is not found or error occures
    '  Application version - HKLM\Software\Motorola\<appName>\CurrentVersion value
    '

    If Not IsNull(appName) And Not IsEmpty(appName) Then
        Dim appVersion

        On Error Resume Next

        appVersion = GetAppProperty(appName, CURRENTVERSION)

        If Not IsNull(appVersion) And Not isEmpty(appVersion) And appVersion <> "" Then
            GetAppVersion = appVersion
        Else
            GetAppVersion = Empty
            Call WriteToLogFile("AppVersion is either Null, Empty or equal to """"", INFO_C)
            Exit Function
        End If

    End If

End Function

Function GetUninstallPath(ByVal appBitInfo)
    ' Return path to the uninstall path in the registry based on the app and system bitness.
    ' [appBitInfo] In - application bitness information.

    Dim processBits, errSource

    On Error Resume Next

    processBits = GetProcessBitInfo()
    errSource = "CommonFunctions::GetUninstallPath: "
    If (processBits = OS_BIT_ERROR) Then
        Err.Raise OS_BIT_ERROR, WScript.ScriptName & "(GetUninstallPath)", errSource & "Unknown processor architecture or error"
        Exit Function
    End If

    If (processBits = appBitInfo) Then
        GetUninstallPath = "Software\Microsoft\Windows\CurrentVersion\Uninstall"
    Else
        GetUninstallPath = "Software\WOW6432Node\Microsoft\Windows\CurrentVersion\uninstall"
    End If

End Function

Function GetAppGUID (ByVal appName, ByVal appBitInfo)
    ' Gets application GUID.
    '
    ' [appName]    In - application name.
    ' [appBitInfo] In - application bitness information.
    '
    ' Return value:
    '  Empty               - if application is not found or error occures
    '  Application version - HKLM\Software\Microsoft\Windows\CurrentVersion\Uninstall key value GUID for application
    '
    ' Remarks:
    '  If return value is Empty, it is recommended to check for errors.

    On Error Resume Next

    ' check for application name
    If Applications.Exists(appName) Then
        ' change application name for compatibility
        appName = Applications(appName)
    End If

    Dim oReg, strKeyPath, strComputer, strSubKey, arrSubKeys(), ret
    Dim processBits, strDisplayName, appGUID, appGUIDVersion
    Dim strKeyProperty, strPropertyValue, strSearchedValue, strDisplayVersion, errSource, strMessage

    'Variables used for storing GUIDs of application installed instances.
    'Eventually, GetAppGUID function should choose and return GUID corresponding to the latest installed version of application.
    Dim appGUIDs, appGUIDsArray, appGUIDsString, iter
    Set appGUIDs = CreateObject("Scripting.Dictionary")

    ' set default return value
    GetAppGUID = Empty

    errSource = "CommonFunctions::GetAppGUID: "

    If Not IsValidString(appName) Then
        strMessage = errSource & "invalid value passed to parameter 'appName'."
        Err.Raise INVALID_FUN_PARAM_ERROR, WScript.ScriptName & "(GetAppGUID)", strMessage
        Exit Function
    End If

    strComputer = "."

    ' get path to the uninstall registry key
    strKeyPath = GetUninstallPath(appBitInfo)

    If (Err.Number <> 0) Then
        Err.Raise Err.Number, Err.Source, Err.Description
        Exit Function
    End If

    Set oReg = GetObject("winmgmts:{impersonationLevel=impersonate}!\\" & strComputer & "\root\default:StdRegProv")

    If (Err.Number <> 0) Then
        Err.Raise Err.Number, Err.Source, errSource & Err.Description
        Exit Function
    End If

    ret = oReg.EnumKey(HKEY_LOCAL_MACHINE, strKeyPath, arrSubKeys)

    If ( (ret <> 0) And (Err.Number <> 0) ) Then
        Err.Raise Err.Number, Err.Source, errSource & Err.Description
        Set oReg = Nothing
        Exit Function
    End If

    UniqueForAppInUnsintallReg appName, strKeyProperty, strSearchedValue

    For Each strSubKey In arrSubKeys
        strPropertyValue = ""
        strDisplayVersion = ""

        ret = oReg.GetStringValue(HKEY_LOCAL_MACHINE, strKeyPath & "\" & strSubKey, strKeyProperty, strPropertyValue)

        If (Err.Number <> 0) And (Err.Number <> -2147217404) Then
            Err.Raise Err.Number, Err.Source, errSource & Err.Description
            Set oReg = Nothing
            Exit Function
        End If

        ' ret equals 0 if property is found in subkey
        If (ret = 0) Then

            ret = InStrRegex(strPropertyValue, strSearchedValue)

            If (Err.Number <> 0) Then
                Err.Raise Err.Number, Err.Source, errSource & Err.Description
                Exit Function
            End If

            ' ret is >= 0 if pattern is found
            If (ret >= 0) Then
                'Check DisplayVersion value for found matching GUID
                ret = oReg.GetStringValue(HKEY_LOCAL_MACHINE, strKeyPath & "\" & strSubKey, "DisplayVersion", strDisplayVersion)

                If (Err.Number <> 0) And (Err.Number <> -2147217404) Then
                    Err.Raise Err.Number, Err.Source, errSource & Err.Description
                    Set oReg = Nothing
                    Exit Function
                End If

                'Add pair GUID=>DisplayVersion to appGUIDs "Scripting.Dictionary" Object (similar to perl hash)
                appGUIDs.Add strSubKey, strDisplayVersion
            End If
        End If
    Next

    Set oReg = Nothing

    'Iterate through all found matching GUIDs - for most of applications inding more than one GUID found is an error
    appGUIDsArray = appGUIDs.Keys
    appGUIDVersion = "0.0.0"
    For iter = 0 To appGUIDs.Count - 1
        'Find GUID corresponding to the latest version of application
        ret = AppVersionComp(appGUIDs(appGUIDsArray(iter)), appGUIDVersion)

        If IsNull(ret) Then
            strMessage = errSource & "unable to check installed version of " & appName & ". GUID: " & appGUIDsArray(iter)
            Err.Raise APP_GUID_ERROR, WScript.ScriptName & "(GetAppGUID)", strMessage
            Exit Function
        ElseIf (ret = 1) Then
            appGUID = appGUIDsArray(iter)
            appGUIDVersion = appGUIDs(appGUIDsArray(iter))
        End If

        'Prepare comma separated string containing list of found GUIDs - used only for error logging purpose
        If Not IsValidString(appGUIDsString) Then
            appGUIDsString = appGUIDsArray(iter)
        Else
            appGUIDsString = appGUIDsString & ", " & appGUIDsArray(iter)
        End If
    Next

    ' Exit and raise error if found more than one instance of application, for which only one instance is allowed
    If ( (appGUIDs.Count > 1) And (Not multipleInstancesApps.Exists(appName)) ) Then
        strMessage = errSource & "more than one instance of application " & appName & " detected. Found GUIDs: " & appGUIDsString
        Err.Raise APP_GUID_ERROR, WScript.ScriptName & "(GetAppGUID)", strMessage
        Exit Function
    End If

    If IsValidString(appGUID) Then
        GetAppGUID = appGUID
    End If

End Function

Function IsAppInstalled (ByVal appName, ByVal appBitInfo)
    ' Returns information about installation status of an application.
    '
    ' [appName]    In - application name.
    ' [appBitInfo] In - application bitness information.
    '
    ' Return value:
    '  True  - application is installed.
    '  False - application is not installed.
    '  Null - error occured. Error should is not propagated to higher level functions.

    ' set default return value
    IsAppInstalled = Null

    Dim appGUID, errSource, strMessage

    On Error Resume Next

    ' check for application name
    If Applications.Exists(appName) Then
        ' change application name for compatibility
        appName = Applications(appName)
    End If

    errSource = "CommonFunctions::IsAppInstalled: "

    If (appName = "Remote Desktop Updates") Then
        IsAppInstalled = IsRemoteDesktopUpdatesInstalled()
    ElseIf (appName = "Internet Explorer") Then
        IsAppInstalled = IsIEReadyToBeInstalled()
    ElseIf (appName = "Microsoft Windows Management Framework") Then
        IsAppInstalled = IsWMFInstalled()
    Else
        ' If GUID for application is found, application is considered as installed
        appGUID = GetAppGUID(appName, appBitInfo)

        ' Check for errors
        strMessage = errSource & " Could not get application GUID. " & " Application name: " & appName & "."
        If (CheckForErrors(strMessage) = True) Then
            IsAppInstalled = Null
            Exit Function
        End If

        If (IsValidString(appGUID)) Then
            IsAppInstalled = True
        Else
            IsAppInstalled = False
        End If
    End If
End Function

Sub UniqueForAppInUnsintallReg (ByVal appName, ByRef strKeyProperty, ByRef strSearchedValue)
    ' This strange sub is called in GetAppGUID function, which obtains
    ' application GUID depending on some value in registry key
    ' HKLM\Software\WOW6432Node\Microsoft\Windows\CurrentVersion\uninstall
    ' UniqueForAppInUnsintallReg role is to fill in two variables: strKeyProperty, strSearchedValue,
    ' which says what property should be test (strKeyProperty) and what is desired value
    ' for application (strSearchedValue)
    '
    ' [appName]    In - application name.
    ' [strKeyProperty] In - property to search in HKLM\Software\WOW6432Node\Microsoft\Windows\CurrentVersion\uninstall\SOME_KEY
    ' [strSearchedValue] In - searched value in HKLM\Software\WOW6432Node\Microsoft\Windows\CurrentVersion\uninstall\SOME_KEY\strKeyProperty
    '

    ' Default values - by default DisplayName should equal appName
    strKeyProperty = "DisplayName"
    strSearchedValue = "^" & appName & "$"

    ' On supplemental CD two products which in fact are Java are provided:
    ' Java - Java family 8, installed in patch-in-place mode
    ' Java - Java family 7, installed in static mode
    ' They can be differentiate by checking their install location
    If (appName = "Java") Then
        strKeyProperty = "InstallLocation"
        strSearchedValue = "Java\\jre8\\"
    ElseIf (appName = "Java_x64") Then
        strKeyProperty = "InstallLocation"
        strSearchedValue = "Java\\jre8\\"
    ElseIf (appName = "Java7") Then
        strKeyProperty = "InstallLocation"
        strSearchedValue = "Java\\jre1\.7"
    ElseIf (appName = "7-Zip") Then
        strSearchedValue = "^" & appName
    ElseIf (appName = "Adobe Reader") Then
        strSearchedValue = "^" & appName
    ElseIf (appName = "Microsoft .NET Framework") Then
        strSearchedValue = "^Microsoft \.NET Framework"
    End If
End Sub

Function InStrRegex (ByVal string1, ByVal pattern)
    ' Returns position in string1 on which pattern is found
    '
    ' [string1]    In - string in which pattern is searched
    ' [pattern]    In - searched pattern
    '
    ' Return value:
    ' Null        - error thrown during execution
    ' -1          - pattern not found in string
    ' number >= 0 - position, on which pattern was found in string1 (0 based)

    Dim Regex, Match, Matches, myPos, strMessage, errSource

    On Error Resume Next

    errSource = "CommonFunctions::InStrRegex: "

    ' Set default value to -1 - pattern not found in string1
    InStrRegex = -1

    Set Regex = New RegExp

    Regex.Global = False               'Set this to false if you just want to find the pattern once
    Regex.IgnoreCase = False
    Regex.Multiline = True
    Regex.Pattern = pattern

    Set Matches = Regex.Execute(string1)

    If (Err.Number <> 0) Then
        Err.Description = errSource & Err.Description
        Err.Raise Err.Number, errSource, Err.Description
        InStrRegex = Null
        Exit Function
    End If

    For Each Match In Matches
        ' Set return value - position on which pattern is found
        InStrRegex = Match.FirstIndex
        Exit For
    Next

End Function

Function IsCurrentUserAdmin
    ' Checks if current process runs as administrator
    '
    ' Return value:
    '  True  - running as administrator.
    '  False - not running as administrator.
    '
    ' Remarks:
    '  based on: http://csi-windows.com/toolkit/csi-isadmin
    Dim ret

    ' set default return value
    IsCurrentUserAdmin = False

    On Error Resume Next

    ret = CreateObject("WScript.Shell").RegRead("HKEY_USERS\S-1-5-19\Environment\TEMP")

    If Err.Number = 0 Then IsCurrentUserAdmin = True

    ' clear error to avoid printing error message if not admin
    Err.Clear
End Function

Function GetOSProperty(ByVal osProperty)
    ' Returns OS version.
    '
    ' Return value:
    '  Empty      - if OS property is not found or error occures.
    '  Property
    '
    ' Remarks:
    '  If return value is Empty, it is recommended to check for errors to make sure that proper information have been retrived.
    Dim strComputer, oWMIService, colOSInfo, oOSProperty, strParameter, errSource

    strComputer = "."

    ' set default return value
    GetOSProperty = Empty

    errSource = "CommonFunctions::GetOSProperty "

    On Error Resume Next

    Set oWMIService = GetObject("winmgmts:\\" & strComputer & "\root\cimv2")

    If (Err.Number <> 0) Then
        Err.Raise Err.Number, Err.Source, Err.Description
        Exit Function
    End If

    Set colOSInfo = oWMIService.ExecQuery("Select * from Win32_OperatingSystem")

    If (Err.Number <> 0) Then
        Err.Raise Err.Number, Err.Source, Err.Description
        Set oWMIService = Nothing
        Exit Function
    End If

    For Each oOSProperty in colOSInfo
        Select Case osProperty
        Case OS_CAPTION
            strParameter = oOSProperty.Caption
            If ( strParameter = "" ) Then
                Err.Raise UNKNOWN_ERROR, WScript.ScriptName & "(GetOSVersion)", "Unknown error"
                Exit Function
            End If
        Case OS_VERSION
            strParameter = oOSProperty.Version
            If ( strParameter = "" ) Then
                Err.Raise UNKNOWN_ERROR, WScript.ScriptName & "(GetOSVersion)", "Unknown error"
                Exit Function
            End If
        Case Else
            Err.Raise UNKNOWN_PARAMETER_ERROR, errSource, "Unknown parameter :" & osProperty
            Exit Function
        End Select
    Next

    GetOSProperty = strParameter

End Function

Function GetOSVersion
    ' Returns OS version.
    '
    ' Return value:
    '  Empty      - if OS version is not found or error occures.
    '  OS version
    '
    ' Remarks:
    '  If return value is Empty, it is recommended to check for errors to make sure that proper information have been retrived.
    '  OS_TYPE_ERROR is raised when OS type error occures.
    '  UNKNOWN_ERROR is raised when an unknown error occures.
    Dim strCaption, osVer

    ' set default return value
    GetOSVersion = Empty

    On Error Resume Next

    strCaption = GetOSProperty(OS_CAPTION)
    If (Err.Number <> 0) Then
        Err.Raise Err.Number, Err.Source, Err.Description
        Exit Function
    End If

    If InStr(1,strCaption, "Vista", vbTextCompare) Then
        osVer = "Vista"
    ElseIf InStr(1,strCaption, "Windows 7", vbTextCompare) Then
        osVer = "Windows7"
    ElseIf InStr(1,strCaption, "2008", vbTextCompare) Then
        osVer = "2008"
    ElseIf InStr(1,strCaption, "XP", vbTextCompare) Then
        osVer = "XP"
    ElseIf InStr(1,strCaption, "2003", vbTextCompare) Then
        osVer = "2003"
    ElseIf InStr(1,strCaption, "2000", vbTextCompare) Then
        osVer = "2000"
    ElseIf InStr(1,strCaption, "Windows 8.1 ", vbTextCompare) Then
        osVer = "Windows_8_1"
    ElseIf InStr(1,strCaption, "Windows 8 ", vbTextCompare) Then
        osVer = "Windows_8"
    ElseIf InStr(1,strCaption, "2012", vbTextCompare) Then
        osVer = "2012"
    Else
        osVer = "__DEF__"
    End If

    If (Err.Number <> 0) Then
        Err.Raise Err.Number, Err.Source, Err.Description
        Exit Function
    Else
        GetOSVersion = osVer
    End If

End Function

Function GetOSVersionNumber
    ' Returns OS version number.
    '
    ' Return value:
    '  Empty      - if OS version is not found or error occures.
    '  OS version number (6.1.1234 like)
    '
    ' Remarks:
    '  If return value is Empty, it is recommended to check for errors to make sure that proper information have been retrived.
    '  UNKNOWN_ERROR is raised when an unknown error occures.
    Dim osVer

    ' set default return value
    GetOSVersionNumber = Empty

    On Error Resume Next

    osVer = GetOSProperty(OS_VERSION)

    If (Err.Number <> 0) Then
        Err.Raise Err.Number, Err.Source, Err.Description
        Exit Function
    End If

    GetOSVersionNumber = osVer
End Function

Function CSI_GetBitness (ByVal Target)
    ' Returns OS bitness.
    '
    ' [Target] In - target.
    '
    ' Return value:
    '  OS bitness
    '  CSI_OS_BIT_ERROR   - if error occures.
    '  CSI_UNKNOWN_ARCH   - if architecture could not been determined.
    '  CSI_UNKNOWN_TARGET - if function is called with unknown target
    '
    ' Remarks:
    '  from http://csi-windows.com/toolkit/csi-getosbits

    On Error Resume Next

    Select Case UCase(Target)
        Case "OS", "WINDOWS", "OPERATING SYSTEM"
            CSI_GetBitness = GetObject("winmgmts:root\cimv2:Win32_Processor='cpu0'").AddressWidth

            ' error handling
            If ((Err.Number = 0) And (CSI_GetBitness <> 32) And (CSI_GetBitness <> 64)) Then
                CSI_GetBitness = CSI_UNKNOWN_ARCH
            ElseIf (Err.Number <> 0) Then
                ' clear error
                Err.Clear
                ' set return value
                CSI_GetBitness = CSI_OS_BIT_ERROR
            End If

        Case "HW", "HARDWARE"
            CSI_GetBitness = GetObject("winmgmts:root\cimv2:Win32_Processor='cpu0'").DataWidth

            ' error handling
            If ((Err.Number = 0) And (CSI_GetBitness <> 32) And (CSI_GetBitness <> 64)) Then
                CSI_GetBitness = CSI_UNKNOWN_ARCH
            ElseIf (Err.Number <> 0) Then
                ' clear error
                Err.Clear
                ' set return value
                CSI_GetBitness = CSI_OS_BIT_ERROR
            End If

        Case "PROCESS", "PROC"
            'One liner to retrieve process architecture string which will reveal If running in 32bit subsystem
            Dim ProcessArch
            ProcessArch = CreateObject("WScript.Shell").Environment("Process")("PROCESSOR_ARCHITECTURE")
            If (Err.Number = 0) Then
                If LCase(ProcessArch) = "x86" Then
                    CSI_GetBitness = 32
                Else
                    If InStr(1,"AMD64,IA64",ProcessArch,1) > 0 Then
                        CSI_GetBitness = 64
                    Else
                        CSI_GetBitness = CSI_UNKNOWN_ARCH 'unknown processor architecture
                    End If
                End If
            Else
                ' clear error
                Err.Clear
                ' set return value
                CSI_GetBitness = CSI_OS_BIT_ERROR
            End If
        Case Else
            CSI_GetBitness = CSI_UNKNOWN_TARGET 'unknown Target item (OS, Hardware, Process)
    End Select

End Function


' It use singleton like idea. Wo don't want to create systemBitsInfo many times as it is very time consuming operation.
Function GetOSBitInfo()
    ' Returns OS bitness.
    '
    ' Return value:
    '  OS bitness
    '  OS_BIT_ERROR   - if error occures.
    '
    ' Remarks:
    '  calls CSI_GetBitness("OS")
    If (IsValidString(systemBitsInfo) = False) Then
        systemBitsInfo = CSI_GetBitness("OS")
    End If

    If ((systemBitsInfo = CSI_UNKNOWN_ARCH) Or (systemBitsInfo = CSI_OS_BIT_ERROR)) Then
        systemBitsInfo = OS_BIT_ERROR
    End If

    GetOSBitInfo = systemBitsInfo
End Function

Function GetProcessBitInfo()
' Returns Process bitness.
    '
    ' Return value:
    '  PROCESS bitness
    '  OS_BIT_ERROR   - if error occures.
    '
    ' Remarks:
    '  calls CSI_GetBitness("PROCESS")
    If (IsValidString(processBitsInfo) = False) Then
        processBitsInfo = CSI_GetBitness("PROCESS")
    End If

    If ((processBitsInfo = CSI_UNKNOWN_ARCH) Or (processBitsInfo = CSI_OS_BIT_ERROR)) Then
        processBitsInfo = OS_BIT_ERROR
    End If

    GetProcessBitInfo = processBitsInfo

End Function

Function GetAppBitInfo(ByVal appName)
    ' return app bitness.
    ' If x64 is found in the end of string, then return 64.
    ' If InStrRegex returns Null, we assume that app is 32 bit.

    Dim strMessage, errSource, ret, osBits

    ret = InStrRegex(appName,"x64$")
    errSource = "CommonFunctions::GetAppBitInfo: "
    osBits = GetOSBitInfo()

    If (IsNull(ret)) Then
        GetAppBitInfo = 32
        strMessage = errSource & "Unable to acquire application bitness - InStrRegex failed. Assuming 32 bit. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, WARNING_C)
        Exit Function
    End If

    'dotNet chooses its installation according to the OS bitness and does not give any indication based in its name
    If ( ( ret >= 0 ) Or ( ( ( appName = "Microsoft .NET Framework" ) Or ( appName = "Microsoft Windows Management Framework" ) ) And ( osBits = 64 ) ) ) Then
        GetAppBitInfo = 64
    Else
        GetAppBitInfo = 32
    End If

End Function

Function IsValidString(ByVal variable)
    ' Check if given variable is initialied.
    ' return True if it is initialized and not empty
    ' return False in other case
    On Error Resume Next

    If IsNull(variable) Or IsEmpty(variable) Or variable = "" Then
        IsValidString = False
    Else
        IsValidString = True
    End If
End Function


Function CheckGlobalVariables()
    ' Check if all necessary global variable are initialized.
    ' return SUCCESS if appName, appMsiVer, appConfigVer, logFileName are initialized and not empty
    ' return FAIL in other case
    CheckGlobalVariables = SUCCESS

    If IsValidString(appName) = False Then
        Call WriteToLogFile("Variable appName is not initialized yet", ERROR_C)
        CheckGlobalVariables = FAIL
        Err.Clear
        Exit Function
    End If

    If IsValidString(appMsiVer) = False Then
        Call WriteToLogFile("Variable appMsiVer is not initialized yet", ERROR_C)
        CheckGlobalVariables = FAIL
        Err.Clear
        Exit Function
    End If

    If IsValidString(appConfigVer) = False Then
        Call WriteToLogFile("Variable appConfigVer is not initialized yet", ERROR_C)
        CheckGlobalVariables = FAIL
        Err.Clear
        Exit Function
    End If

    If IsValidString(logFileName) = False Then
        Call WriteToLogFile("Variable logFileName is not initialized yet", ERROR_C)
        CheckGlobalVariables = FAIL
        Err.Clear
        Exit Function
    End If

End Function

Function IsMotorolaJavaRunning()
    '
    ' [titleName]  In - "string" Windows title bar process name to check.
    '
    ' Return value:
    '  TRUE - process is running
    '  FALSE - process is not running
    '
    On Error Resume Next

	IsMotorolaJavaRunning = IsRunningByPath(PROCESS, arrMotorolaJavas)

End Function

Function IsRunningByName (ByVal oType, ByRef name)
    '
    ' [oType] In - "object" (process/service/script) type.
    ' [name]  In - "object" (process/service/script) name to check. Can be string or array.
    '
    ' Return value:
    '  TRUE - process/service is running
    '  FALSE - process/service is not running
    '
    ' Remarks:
    '  Function can raise na error, so it is necessary to check for errors after calling it.
    Dim oWmi, obj, objList, strComputer, strMessage, errSource, runningNum, namesToCheck, condition, iter

    On Error Resume Next

    ' Set default return value
    IsRunningByName = FALSE

    errSource = "CommonFunctions::IsRunningByName "

    'Validate function parameters
    If Not (oType = PROCESS Or oType = SERVICE Or oType = SCRIPT) Then
        strMessage = errSource & "Wrong value of parameter oType. Valid values are 'PROCESS' or 'SERVICE' or 'SCRIPT'."
        Err.Raise INVALID_FUN_PARAM_ERROR, errSource, strMessage
        Exit Function
    End If


    If (IsArray(name) = True) Then
        For iter = 0 To UBound(name)
            If Not IsValidString(name(iter)) Then
                strMessage = errSource & "At least one entry in 'name' array is Null or Empty." & " Application name: " & appName
                Err.Raise INVALID_FUN_PARAM_ERROR, errSource, strMessage
                Exit Function
            End If
            If (oType = SCRIPT) Then
                condition = "CommandLine Like '%" & name(iter) & "%'"
            Else
                condition = "Name = '" & name(iter) & "'"
            End If
            If (iter = 0) Then
                namesToCheck = condition
            Else
                namesToCheck = namesToCheck & " OR " & condition
            End If
        Next
    Else
        If Not IsValidString(name) Then
            strMessage = errSource & "Null or Empty value passed to parameter 'name'."
            Err.Raise INVALID_FUN_PARAM_ERROR, errSource, strMessage
            Exit Function
        End If

        If (oType = SCRIPT) Then
            namesToCheck = "CommandLine Like '%" & name & "%'"
        Else
            namesToCheck = "Name = '" & name & "'"
        End If
    End If

    'Parameters are valid here, check if process/service is running
    strComputer = "."

    Set oWmi = GetObject("winmgmts:{impersonationLevel=impersonate}!\\" & strComputer & "\root\cimv2")

    If (Err.Number <> 0) Then
        Err.Description = errSource & " Could not create WMI object. "
        Err.Raise Err.Number, errSource, Err.Description
        Exit Function
    End If

    If ((oType = PROCESS) Or (oType = SCRIPT)) Then
        ' get all running matching processes by name
        Set objList = oWmi.ExecQuery("SELECT * FROM Win32_Process WHERE (" & namesToCheck & ")")
    ElseIf (oType = SERVICE) Then
        ' get all running matching services by name
        Set objList = oWmi.ExecQuery("SELECT * FROM Win32_Service WHERE (" & namesToCheck & ") AND State = 'Running'")
    End If

    ' Get number of running instances of searched process/service
    runningNum = objList.Count

    ' Check for errors. If command called in ExecQuery is wrong, object is returned and no error is raised.
    ' Error is raised upon attempt to make operation on this faulty object (e.g. method .Count)
    If (Err.Number <> 0) Then
        Err.Description = errSource & "Could not list matching processes/services. " & Err.Description
        Err.Raise Err.Number, errSource, strMessage
        Exit Function
    End If

    'Return True if number of running searched process/service instances is greater than 0
    If (runningNum > 0) Then
        IsRunningByName = TRUE
    End If


End Function

Function IsRunningByPath (ByVal oType, ByRef path)
    '
    ' [oType] In - "object" (process/service/script) type.
    ' [path]  In - "object" (process/service/script) path to check. Can be string or array.
    '
    ' Return value:
    '  TRUE - process/service is running
    '  FALSE - process/service is not running
    '
    ' Remarks:
    '  Function can raise na error, so it is necessary to check for errors after calling it.
    Dim oWmi, obj, objList, strComputer, strMessage, errSource, runningNum, namesToCheck, condition, iter

    On Error Resume Next

    ' Set default return value
    IsRunningByPath = FALSE

    errSource = "CommonFunctions::IsRunningByPath "

    'Validate function parameters
    If Not (oType = PROCESS Or oType = SERVICE Or oType = SCRIPT) Then
        strMessage = errSource & "Wrong value of parameter oType. Valid values are 'PROCESS' or 'SERVICE' or 'SCRIPT'."
        Err.Raise INVALID_FUN_PARAM_ERROR, errSource, strMessage
        Exit Function
    End If


    If (IsArray(path) = True) Then
        For iter = 0 To UBound(path)
            If Not IsValidString(path(iter)) Then
                strMessage = errSource & "At least one entry in 'path' array is Null or Empty." & " Application name: " & appName
                Err.Raise INVALID_FUN_PARAM_ERROR, errSource, strMessage
                Exit Function
            End If
            If (oType = SCRIPT) Then
                condition = "CommandLine Like '%" & path(iter) & "%'"
            Else
                condition = "ExecutablePath = '" & path(iter) & "'"
            End If
            If (iter = 0) Then
                namesToCheck = condition
            Else
                namesToCheck = namesToCheck & " OR " & condition
            End If
        Next
    Else
        If Not IsValidString(path) Then
            strMessage = errSource & "Null or Empty value passed to parameter 'path'."
            Err.Raise INVALID_FUN_PARAM_ERROR, errSource, strMessage
            Exit Function
        End If

        If (oType = SCRIPT) Then
            namesToCheck = "CommandLine Like '%" & path & "%'"
        Else
            namesToCheck = "ExecutablePath = '" & path & "'"
        End If
    End If

    'Parameters are valid here, check if process/service is running
    strComputer = "."

    Set oWmi = GetObject("winmgmts:{impersonationLevel=impersonate}!\\" & strComputer & "\root\cimv2")

    If (Err.Number <> 0) Then
        Err.Description = errSource & " Could not create WMI object. "
        Err.Raise Err.Number, errSource, Err.Description
        Exit Function
    End If

    If ((oType = PROCESS) Or (oType = SCRIPT)) Then
        ' get all running matching processes by path
        Set objList = oWmi.ExecQuery("SELECT * FROM Win32_Process WHERE " & namesToCheck & "")
    ElseIf (oType = SERVICE) Then
        ' get all running matching services by path
        Set objList = oWmi.ExecQuery("SELECT * FROM Win32_Service WHERE (" & namesToCheck & ") AND State = 'Running'")
    End If

    ' Get number of running instances of searched process/service
    runningNum = objList.Count

    ' Check for errors. If command called in ExecQuery is wrong, object is returned and no error is raised.
    ' Error is raised upon attempt to make operation on this faulty object (e.g. method .Count)
    If (Err.Number <> 0) Then
        Err.Description = errSource & "Could not list matching processes/services. " & Err.Description
        Err.Raise Err.Number, errSource, strMessage
        Exit Function
    End If

    'Return True if number of running searched process/service instances is greater than 0
    If (runningNum > 0) Then
        IsRunningByPath = TRUE
    End If

End Function

Function IsTaskScheduled (ByVal name)
    '
    ' [name]  In - task name to check.
    '
    ' Return value:
    '  TRUE - task is scheduled
    '  FALSE - task is not scheduled
    '
    ' Remarks:
    '  Function can raise na error, so it is necessary to check for errors after calling it.
    Dim ret, strMessage, strCommand, errSource, WshShell

    On Error Resume Next

    ' Set default return value
    IsTaskScheduled = FALSE

    errSource = "CommonFunctions::IsTaskScheduled "

    'Validate function parameters
    If Not IsValidString(name) Then
        strMessage = errSource & "Null or Empty value passed to parameter 'name'."
        Err.Raise INVALID_FUN_PARAM_ERROR, errSource, strMessage
        Exit Function
    End If

    'Parameters are valid here, check if process/service is running or task is scheduled

    ' Check if searched task is scheduled
    Set WshShell = CreateObject("Wscript.Shell")

    If (Err.Number <> 0) Then
        Err.Description = errSource & " Could not create shell. " & Err.Description
        Err.Raise Err.Number, errSource, Err.Description
        Exit Function
    End If

    strCommand = "SCHTASKS /query /TN " & name
    ret = WshShell.Run(strCommand, 0, TRUE)

    If (Err.Number <> 0) Then
        Err.Description = errSource & " Error occured while running command: " & strCommand & ". " & Err.Description
        Err.Raise Err.Number, errSource, strMessage
        Exit Function
    End If

    'Return True (task is scheduled)
    If (ret = 0) Then
        IsTaskScheduled = TRUE
    End If
End Function

Function DeleteTask (ByVal name)
    '
    ' [name]  In - task name to check.
    '
    ' Return value:
    '  TRUE - task is scheduled
    '  FALSE - task is not scheduled
    '
    ' Remarks:
    '  Function can raise na error, so it is necessary to check for errors after calling it.
    Dim strMessage, strCommand, errSource, WshShell, ret

    On Error Resume Next

    ' Set default return value
    DeleteTask = FAIL

    errSource = "CommonFunctions::DeleteTask "

    'Validate function parameters
    If Not IsValidString(name) Then
        strMessage = errSource & "Null or Empty value passed to parameter 'name'."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    'Parameters are valid here, check if process/service is running or task is scheduled

    ' Check if searched task is scheduled
    Set WshShell = CreateObject("Wscript.Shell")

    strMessage = errSource & " Could not create shell. " & " Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    strCommand = "SCHTASKS /delete /TN " & name & " /F"
    WshShell.Run strCommand, 0, TRUE

    strMessage = errSource & " Error occured while running command: " & strCommand & "." & " Application name: "  & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ret = IsTaskScheduled(name)

    strMessage = errSource & "failed." & " Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If (ret = TRUE) Then
        strMessage = errSource & "Unexpectedly task is still scheduled." & " Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
    ElseIf (ret = FALSE) Then
        ' This should be the only place, where return value is Set to SUCCESSS
        DeleteTask = SUCCESS
    Else
        strMessage = errSource & "Function IsTaskScheduled returned unexpected value." & " Application name: " & appName & ". Error number is " & Err.Number & ": " & Err.Description
        Call WriteToLogFile(strMessage, ERROR_C)
    End If
End Function

Function StopByName (ByVal oType, ByRef name, ByVal timeout)
    ' Stops process/service.
    '
    ' [oType] In - "object" (process/service/script) type.
    ' [name]  In - "object" (process/service/script) name to stop. Can be string or array.
    ' [timeout] In - If process/service remains active after timeout miliseconds, function exits with FAIL
    '
    ' Return value:
    '  SUCCESS - process/service stopped or not found/available.
    '  FAIL - unable to stop process/service or process/service name is null or empty or error occures.
    '
    ' Remarks:
    '  If return value is FAIL, it is recommended to check for errors to make sure that function has been properly executed.
    Dim oWmi, obj, objList, strComputer, strMessage, errSource, ret, runningNum, namesToStop, condition, iter

    On Error Resume Next

    ' Set default return value
    StopByName = FAIL

    errSource = "CommonFunctions::StopByName "

    'Validate function parameters
    If Not (oType = PROCESS Or oType = SERVICE Or oType = SCRIPT) Then
        strMessage = errSource & "Wrong value of parameter oType. Valid values are 'PROCESS' or 'SCRIPT' or 'SERVICE'." & " Application name: " & appName
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    If (IsArray(name) = True) Then

        For iter = 0 To UBound(name)
            If Not IsValidString(name(iter)) Then
                strMessage = errSource & "At least one entry in 'name' array is Null or Empty." & " Application name: " & appName
                Call WriteToLogFile(strMessage, ERROR_C)
                Exit Function
            End If
            If (oType = SCRIPT) Then
                condition = "CommandLine Like '%" & name(iter) & "%'"
            Else
                condition = "Name = '" & name(iter) & "'"
            End If
            If (iter = 0) Then
                namesToStop = condition
            Else
                namesToStop = namesToStop & " OR " & condition
            End If
        Next

    Else
        If Not IsValidString(name) Then
            strMessage = errSource & "Null or Empty value passed to parameter 'name'." & " Application name: " & appName
            Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If

        If (oType = SCRIPT) Then
            namesToStop = "CommandLine Like '%" & name & "%'"
        Else
            namesToStop = "Name = '" & name & "'"
        End If
    End If

    'Parameters are valid here, check if process/service is running
    strComputer = "."
    Set oWmi = GetObject("winmgmts:{impersonationLevel=impersonate, (Debug)}!\\" & strComputer & "\root\cimv2")

    strMessage = errSource & "Could not create WMI object." & " Application name: " & appName & "."

    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If ((oType = PROCESS) Or (oType = SCRIPT)) Then
        ' get all running matching processes by name
        Set objList = oWmi.ExecQuery("SELECT * FROM Win32_Process WHERE (" & namesToStop & ")")
    ElseIf (oType = SERVICE) Then
        ' get all running matching services by name
        Set objList = oWmi.ExecQuery("SELECT * FROM Win32_Service WHERE (" & namesToStop & ") AND State = 'Running'")
    End If

    ' Get number of running instances of searched process/service
    runningNum = objList.Count

    ' Check for errors. If command called in ExecQuery is wrong, object is returned and no error is raised.
    ' Error is raised upon attempt to make operation on this faulty object (e.g. method .Count)
    strMessage = errSource & "Could not list matching processes/services." & " Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    'Return SUCCESS if matching process/service is not running - there is no need to stop anything
    If (runningNum = 0) Then
        StopByName = SUCCESS
        Exit Function
    End If

    'If program reaches this point, there process/serivce is running and we need to stop it.
    For Each obj in objList
        If ((oType = PROCESS) or (oType = SCRIPT)) Then
            ' terminate process
            obj.Terminate()
        ElseIf (oType = SERVICE) Then
            ' stop service
            obj.StopService()
        End If

        strMessage = errSource & "Error occured while trying to stop process/service." & " Application name: " & appName & "."
        If (CheckForErrors(strMessage) = True) Then
            Exit Function
        End If
    Next

    ' Check here if process/service/script was really stopped.
    Dim stopTime, timeoutInSeconds
    stopTime = Timer()
    timeoutInSeconds = timeout / 1000

    Do While ( (Timer() - stopTime) < timeoutInSeconds )
    ret = IsRunningByName(oType, name)

    strMessage = errSource & "failed." & " Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If (ret = TRUE) Then
            ' Process or service is still running. Sleep 100 miliseconds before another check.
            Wscript.Sleep(100)
        ElseIf (ret = FALSE) Then
            ' This should be the only place, where return value is Set to SUCCESS
            StopByName = SUCCESS
            Exit Function
    Else
        strMessage = errSource & "Function IsRunningByName returned unexpected value. " & " Application name: " & appName & ". Error number is " & Err.Number & ": " & Err.Description
        Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If
    Loop

    'IsRunningByName returned true
    strMessage = errSource & "Unexpectedly process/service is still running. " & " Application name: " & appName
    Call WriteToLogFile(strMessage, ERROR_C)

End Function

Function StopMotorolaJavasByPath ()
    ' Stops all Motorola Java processes.
    '
    ' Return value:
    '  SUCCESS - Motorola Java processes stopped or not found/available.
    '  FAIL - unable to stop Motorola Java processes or processes path is null or empty or error occurs.
    '
    On Error Resume Next

    StopMotorolaJavasByPath = StopByPath(PROCESS, arrMotorolaJavas, JAVA_DEFAULT_STOP_TIMEOUT)

End Function

Function StopMotorolaOpenJDKByPath ()
    ' Stops Motorola OpenJDK processes.
    '
    ' Return value:
    '  SUCCESS - Motorola OpenJDK processes stopped or not found/available.
    '  FAIL - unable to stop Motorola OpenJDK processes or processes path is null or empty or error occurs.
    '
    ' Remarks:
    '  If return value is FAIL, it is recommended to check for errors to make sure that function has been properly executed.
    Dim appsToStop(2)

    On Error Resume Next

    ' Set default return value
    StopMotorolaOpenJDKByPath = FAIL

    appsToStop(0) = "C:\\Program Files (x86)\\Motorola\\Motorola OpenJDK\\bin\\javaw.exe"
    appsToStop(1) = "C:\\Program Files (x86)\\Motorola\\Motorola OpenJDK\\bin\\java.exe"
	appsToStop(2) = "C:\\Program Files (x86)\\Motorola\\Motorola OpenJDK\\bin\\keytool.exe"

    StopMotorolaOpenJDKByPath = StopByPath(PROCESS, appsToStop, JAVA_DEFAULT_STOP_TIMEOUT)

End Function


Function StopByPath (ByVal oType, ByRef path, ByVal timeout)
    ' Stops process/service.
    '
    ' [oType] In - "object" (process/service/script) type.
    ' [path]  In - "object" (process/service/script) path to stop. Can be string or array.
    ' [timeout] In - If process/service remains active after timeout miliseconds, function exits with FAIL
    '
    ' Return value:
    '  SUCCESS - process/service stopped or not found/available.
    '  FAIL - unable to stop process/service or process/service path is null or empty or error occures.
    '
    ' Remarks:
    '  If return value is FAIL, it is recommended to check for errors to make sure that function has been properly executed.
    Dim oWmi, obj, objList, strComputer, strMessage, errSource, ret, runningNum, namesToStop, condition, iter

    On Error Resume Next

    ' Set default return value
    StopByPath = FAIL

    errSource = "CommonFunctions::StopByPath "

    'Validate function parameters
    If Not (oType = PROCESS Or oType = SERVICE Or oType = SCRIPT) Then
        strMessage = errSource & "Wrong value of parameter oType. Valid values are 'PROCESS' or 'SCRIPT' or 'SERVICE'." & " Application name: " & appName
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    If (IsArray(path) = True) Then

        For iter = 0 To UBound(path)
            If Not IsValidString(path(iter)) Then
                strMessage = errSource & "At least one entry in 'path' array is Null or Empty." & " Application name: " & appName
                Call WriteToLogFile(strMessage, ERROR_C)
                Exit Function
            End If
            If (oType = SCRIPT) Then
                condition = "CommandLine Like '%" & path(iter) & "%'"
            Else
                condition = "ExecutablePath = '" & path(iter) & "'"
            End If
            If (iter = 0) Then
                namesToStop = condition
            Else
                namesToStop = namesToStop & " OR " & condition
            End If
        Next

    Else
        If Not IsValidString(path) Then
            strMessage = errSource & "Null or Empty value passed to parameter 'path'." & " Application name: " & appName
            Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If

        If (oType = SCRIPT) Then
            namesToStop = "CommandLine Like '%" & path & "%'"
        Else
            namesToStop = "ExecutablePath = '" & path & "'"
        End If
    End If

    'Parameters are valid here, check if process/service is running
    strComputer = "."
    Set oWmi = GetObject("winmgmts:{impersonationLevel=impersonate, (Debug)}!\\" & strComputer & "\root\cimv2")

    strMessage = errSource & "Could not create WMI object." & " Application name: " & appName & "."

    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If ((oType = PROCESS) Or (oType = SCRIPT)) Then
        ' get all running matching processes by path
        Set objList = oWmi.ExecQuery("SELECT * FROM Win32_Process WHERE " & namesToStop & "")
    ElseIf (oType = SERVICE) Then
        ' get all running matching services by path
        Set objList = oWmi.ExecQuery("SELECT * FROM Win32_Service WHERE (" & namesToStop & ") AND State = 'Running'")
    End If

    ' Get number of running instances of searched process/service
    runningNum = objList.Count

    ' Check for errors. If command called in ExecQuery is wrong, object is returned and no error is raised.
    ' Error is raised upon attempt to make operation on this faulty object (e.g. method .Count)
    strMessage = errSource & "Could not list matching processes/services." & " Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    'Return SUCCESS if matching process/service is not running - there is no need to stop anything
    If (runningNum = 0) Then
        StopByPath = SUCCESS
        Exit Function
    End If

    'If program reaches this point, there process/serivce is running and we need to stop it.
    For Each obj in objList
        If ((oType = PROCESS) or (oType = SCRIPT)) Then
            ' terminate process
            obj.Terminate()
        ElseIf (oType = SERVICE) Then
            ' stop service
            obj.StopService()
        End If

        strMessage = errSource & "Error occured while trying to stop process/service." & " Application name: " & appName & "."
        If (CheckForErrors(strMessage) = True) Then
            Exit Function
        End If
    Next

    ' Check here if process/service/script was really stopped.
    Dim stopTime, timeoutInSeconds
    stopTime = Timer()
    timeoutInSeconds = timeout / 1000

    Do While ( (Timer() - stopTime) < timeoutInSeconds )
    ret = IsRunningByPath(oType, path)

    strMessage = errSource & "failed." & " Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If (ret = TRUE) Then
		' Process or service is still running. Sleep 100 miliseconds before another check.
		Wscript.Sleep(100)
	ElseIf (ret = FALSE) Then
		' This should be the only place, where return value is Set to SUCCESS
		StopByPath = SUCCESS
		Exit Function
    Else
        strMessage = errSource & "Function IsRunningByPath returned unexpected value. " & " Application name: " & appName & ". Error number is " & Err.Number & ": " & Err.Description
        Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If
    Loop

    'IsRunningByPath returned true
    strMessage = errSource & "Unexpectedly process/service is still running. " & " Application name: " & appName
    Call WriteToLogFile(strMessage, ERROR_C)

End Function

Function UpgradeKeys(ByVal appName, ByVal oldVersion)
    ' function updates all necessary keys during upgrade.
    ' [appName] In - application name.
    ' [oldVersion]  - previous version of the installed application
    ' return SUCCESS if all keys are updated.
    ' return FAIL if an error pop up.
    Dim strMessage
    UpgradeKeys = SUCCESS

    On Error Resume Next

    Call WriteToLogFile("Running UpgradeKeys procedure", INFO_C)

    If(CheckForErrors("UpgradeKeys fails for " & appName & "It was not able to write to log file") = True) Then
        UpgradeKeys = FAIL
        Exit Function
    End If

    UpgradeKeys = UpdateAppProperty(appName, CURRENTVERSION, appMsiVer)
    If(UpgradeKeys = FAIL) Then
        Exit Function
    End If

    UpgradeKeys = UpdateAppProperty(appName, PREVIOUSVERSION, oldVersion)
    If(UpgradeKeys = FAIL) Then
        Exit Function
    End If

    UpgradeKeys = UpdateAppProperty(appName, UPGRADETIME, CurrentTimeGeneralDateStr())
    If(UpgradeKeys = FAIL) Then
        Exit Function
    End If


End Function

Function UpgradeGUID (ByVal appName)
    ' Upgrade GUID in win registry (according to SYS-033).
    ' [appName] In - application name.

    Dim guidVal, strMessage

    UpgradeGUID = SUCCESS

    On Error Resume Next

    If IsAppInstalled(appName, GetAppBitInfo(appName)) = True Then

        guidVal = GetAppGUID(appName, GetAppBitInfo(appName))
        strMessage = "Can't get GUID for " & appName & " " & Err.Description
        If(CheckForErrors(strMessage) = True) Then
            UpgradeGUID = FAIL
            Exit Function
        End If

        If(UpdateAppProperty(appName, GUID, guidVal) = FAIL) Then
            UpgradeGUID = FAIL
            Exit Function
        End If
    End If

    strMessage = "Could not check installation status for " & appName & "."
    ' check for errors
    If(CheckForErrors(strMessage) = True) Then
        UpgradeGUID = FAIL
        Exit Function
    End If
End Function

Function FixThirdpartyRegistryKeys(ByVal appName)
    ' Initialize all necessary keys for third party tools
    ' It updates CurrentVersion key with the already installed version. It check application registry keys.
    ' It is run only for Java's, Adobe Reader and 7-Zip.
    ' The reason is that we want to detect the case when newer than expected soft in installed.
    ' [appName] In - application name.
    Dim guidVal, uninstallPath, versionPath, installedVersion, appVersion, appVerComp

    FixThirdpartyRegistryKeys = SUCCESS

    On Error Resume Next
    guidVal = GetAppGUID(appName, GetAppBitInfo(appName))
    If (Err.Number <> 0) Then
        Call WriteToLogFile("Cannot find GUID of installed application: " & appName & ". " & Err.Description, ERROR_C)
        FixThirdpartyRegistryKeys = FAIL
        Err.Clear
        Exit Function
    End If

    uninstallPath = GetUninstallPath(GetAppBitInfo(appName))
    If (Err.Number <> 0) Then
        Call WriteToLogFile("Cannot read uninstall path for " & appName & ". " & Err.Description, ERROR_C)
        FixThirdpartyRegistryKeys = FAIL
        Err.Clear
        Exit Function
    End If

    versionPath = "HKLM\" & uninstallPath & "\" & guidVal & "\DisplayVersion"
    installedVersion = GetProperty(versionPath)
    If(Err.Number <> 0) Then
        Err.Clear
        FixThirdpartyRegistryKeys = FAIL
        Exit Function
    End If
    appVersion = GetAppVersion(appName)

    If (Not IsEmpty(appVersion)) Then
        appVerComp = AppVersionComp(appVersion, installedVersion)
        If appVerComp <> 0 Then
            Call WriteToLogFile("The following version is installed " & installedVersion & " and CurrentVersion registry key refers to " & appVersion, INFO_C)
            If(Err.Number <> 0) Then
                Err.Clear
                FixThirdpartyRegistryKeys = FAIL
                Exit Function
            End If
            If(UpdateAppProperty(appName, CURRENTVERSION, installedVersion) = FAIL) Then
                FixThirdpartyRegistryKeys = FAIL
                Exit Function
            End If
        End If
    Else
        Call WriteToLogFile("The following version is installed " & installedVersion & " and CurrentVersion registry key is not set yet", INFO_C)
        If(UpdateAppProperty(appName, CURRENTVERSION, installedVersion) = FAIL) Then
            FixThirdpartyRegistryKeys = FAIL
            Exit Function
        End If
    End If

End Function

Function FixRegistryKeys(ByVal appName, ByVal isInstalled)
    ' Initialize all necessary keys
    ' [appName] In - application name.
    ' [isInstalled] In - if app is already installed.

    FixRegistryKeys = SUCCESS

    On Error Resume Next

    Call WriteToLogFile("Running FixRegistryKeys procedure", INFO_C)
    If(Err.Number <> 0) Then
        Err.Clear
        FixRegistryKeys = FAIL
        Exit Function
    End If

    ' if app is installed the following keys should be set: CURRENTVERSION
    If((Applications.Exists(appName) = True) and (isInstalled = True)) Then
        FixRegistryKeys = FixThirdpartyRegistryKeys(appName)
    ElseIf(isInstalled = True) Then
    If(InitializeSingleKey(appName, CURRENTVERSION, "0.0.0") = FAIL) Then
        FixRegistryKeys = FAIL
        Exit Function
    End If
    end If


End Function

Function SetInitialKeys(ByVal appName)
    ' function set all keys required when application is installed.
    ' It is used when fresh we run install scenerio.
    ' [appName] In - application name.
    ' return SUCCESS if all necessary keys are set or they were already present.
    ' return FAIL if an error pop up during seting key registry value
    Dim strMessage

    SetInitialKeys = SUCCESS

    On Error Resume Next

    Call WriteToLogFile("Running SetInitialKeys procedure", INFO_C)
    If(Err.Number <> 0) Then
        SetInitialKeys = FAIL
        Err.Clear
        Exit Function
    End If

    ' if app is installed the following keys should be set: INITIALVERSION, INITIALTIME, CURRENTVERSION
    If(InitializeSingleKey(appName, INITIALVERSION, appMsiVer) = FAIL) Then
        SetInitialKeys = FAIL
        Exit Function
    End If
    If(InitializeSingleKey(appName, INITIALTIME, CurrentTimeGeneralDateStr()) = FAIL) Then
        SetInitialKeys = FAIL
        Exit Function
    End If

    If(InitializeSingleKey(appName, CURRENTVERSION, appMsiVer) = FAIL) Then
        SetInitialKeys = FAIL
        Exit Function
    End If

    If(InitializeSingleKey(appName, PREVIOUSVERSION, "") = FAIL) Then
        SetInitialKeys = FAIL
        Exit Function
    End If

    If(InitializeSingleKey(appName, UPGRADETIME, "") = FAIL) Then
        SetInitialKeys = FAIL
        Exit Function
    End If

    strMessage = "SetInitialKeys fails for " & appName
    ' check for errors
    If(CheckForErrors(strMessage) = True) Then
        SetInitialKeys = FAIL
        Exit Function
    End If

End Function

Function InitializeSingleKey(ByVal appName, ByVal propName, ByVal propValue)
    ' if the given key does not exists it is added and value is set to propValue
    ' if the key already exists nothing happen.
    ' [appName] In - application name.
    ' [propName] In - property name.
    ' [propValue]   In - property value.
    ' return SUCCESS operation does not throw any error.
    ' return FAIL if an error pop

    Dim strMessage
    On Error Resume Next

    InitializeSingleKey = SUCCESS

    If (IsNull(GetAppProperty(appName, propName)) = True) Then
        If(UpdateAppProperty(appName, propName, propValue) = FAIL) Then
            Call WriteToLogFile("Can't initialize " & propName & " for " & appName, INFO_C)
            InitializeSingleKey = FAIL
            Exit Function
        End If
    End If
End Function

Function GetAppProperty(ByVal appName, ByVal propName)
    ' return value if reg key exists
    ' [appName] In - application name.
    ' [propName] In - property name.
    ' return Null if reg key does not exist

    GetAppProperty = GetProperty(GetPropertyPath(appName, propName))

End Function

Function GetProperty(ByVal strPropPath)
    ' return value if reg key exists
    ' [strPropPath] path to the key registry
    ' return Null if reg key does not exist
    On Error Resume Next
    GetProperty = CreateObject("WScript.Shell").RegRead(strPropPath)
    If Err.Number <> 0 Then
        Call WriteToLogFile("Can't read path " & strPropPath & " for " & appName & " " & Err.Description, INFO_C)
        Err.Clear
        GetProperty = Null
    End If
End Function

Function DeleteAppProperty(ByVal appName, ByVal propName)
    ' delete req property/value under reg key appName
    ' [appName] In - application name.
    ' [propName] In - property name.
    ' return True if property does not exists or was removed.
    ' return False if property exists and it fails to remove it.

    On Error Resume Next

    DeleteAppProperty = True

    If Not IsNull(GetAppProperty(appName, propName)) Then
        CreateObject("WScript.Shell").RegDelete(GetPropertyPath(appName, propName))
        If (Err.Number <> 0) Then
            Call WriteToLogFile("Can't delete following property: " & propName & " for " & appName, ERROR_C)
            Err.Clear
            DeleteAppProperty = False
            Exit Function
        End If
    End If

End Function


Function UpdateAppProperty (ByVal appName, ByVal propName, ByVal regVal)
    ' Updates property value under given registry key for application.
    ' Creates property if not exists.
    '
    ' [appName]  In - application name.
    ' [propName] In - property name.
    ' [regVal]   In - property value.
    '
    ' return SUCCESS operation does not throw any error.
    ' return FAIL if an error pop


    Dim strPropPath, strMessage

    UpdateAppProperty = SUCCESS

    If IsValidString(appName) = True And IsValidString(propName) = True And Not IsNull(regVal) And Not IsEmpty(regVal) Then

        On Error Resume Next

        ' set property path
        strPropPath = GetPropertyPath(appName, propName)
        If IsNull(strPropPath) Then
            strMessage = "Could not update/create the following key: " & propName & " for " & appName & ". Invalid propertyPath "
            Call WriteToLogFile(strMessage, ERROR_C)
            Err.Clear
            UpdateAppProperty = FAIL
        Else

            If IsNumeric(regVal) Then
                ' write DWORD value to the property
                CreateObject("WScript.Shell").RegWrite strPropPath, regVal, "REG_DWORD"
            Else
                ' write String value to the property
                CreateObject("WScript.Shell").RegWrite strPropPath, regVal
            End If

            If (Err.Number <> 0) Then
                strMessage = "Could not update/create the following key: " & propName & " for " & appName & ". Error number is " & Err.Number & ": " & Err.Description
                Call WriteToLogFile(strMessage, ERROR_C)
                Err.Clear
                UpdateAppProperty = FAIL
            End If
        End If
    Else
        strMessage = "Could not update/create the following key: " & propName & " for " & appName & ". Invalid parameters "
        Call WriteToLogFile(strMessage, ERROR_C)
        Err.Clear
        UpdateAppProperty = FAIL
    End If

End Function

Function GetRegSoftware(ByVal appName)
    ' return the path depending on the bitness
    ' [appName]  In - application name.
    ' return path to the registry key
    ' return Null if an error pop up.
    Dim processBits
    ' default value
    On Error Resume Next
    GetRegSoftware = Null
    processBits = GetProcessBitInfo()
    If (processBits = OS_BIT_ERROR) Then
        Call WriteToLogFile("Cannot get software registry because of problems with getting information about processBits", ERROR_C)
        Exit Function
    End If

    If (processBits = GetAppBitInfo(appName)) Then
       GetRegSoftware = REG_SOFTWARE
    Else
       GetRegSoftware = REG_SOFTWARE_WOW6432NODE
    End If
End Function

Function GetAppProgramFiles(ByVal appName)
    ' return the Program Fiels path depending on the bitness
    ' [appName]  In - application name.
    ' return path to Program Files folder
    ' return Null if an error pop up.
    Dim osBits
    ' default value
    On Error Resume Next
    GetAppProgramFiles = Null
    osBits = GetOSBitInfo()
    If (osBits = OS_BIT_ERROR) Then
        Call WriteToLogFile("Cannot get Program Files folder location because of problems with getting information about OS bitness", ERROR_C)
        Exit Function
    End If

    If (osBits = GetAppBitInfo(appName)) Then
       GetAppProgramFiles = FOLDER_PROGRAM_FILES
    Else
       GetAppProgramFiles = FOLDER_PROGRAM_FILES_X86
    End If
End Function

Function GetPropertyPath(ByVal appName, ByVal propName)
    ' return the path depending on the bitness
    ' [appName]  In - application name.
    ' [propName] In - property name.
    ' return path to the registry key
    ' return Null if an error pop up.
    Dim processBits, regSoftware
    On Error Resume Next
    ' default value
    GetPropertyPath = Null
    regSoftware = GetRegSoftware(appName)
    If (IsNull(regSoftware)) Then
        Call WriteToLogFile("Cannot determine property path", ERROR_C)
        Exit Function
    End If

    GetPropertyPath = "HKLM\" & regSoftware & "\Motorola\" & appName & "\" & propName
End Function

 Function CheckForErrors(ByVal strMessage)
    ' check if no errors occur. If the error is found this function will clear Err object
    ' return True if an error occur
    ' return False if no error was found
    CheckForErrors = False
    If (Err.Number <> 0) Then
        CheckForErrors = True
        Call WriteToLogFile(strMessage & " Error number is " & Err.Number & ": " & Err.Description, ERROR_C)
        Err.Clear
        Exit Function
    End If
End Function

Function AppVersionComp (ByVal appVer1, ByVal appVer2)
    ' Compares two application versions and returns a value that represents the result of the comparison.
    '
    ' [appVer1] In - application version 1 <major version>.<minor version>.<build number>
    ' [appVer2] In - application version 2 <major version>.<minor version>.<build number>
    ' OR
    ' [appVer1] In - application version 1 <major version>.<minor version>.<build number>.<4th element>
    ' [appVer2] In - application version 2 <major version>.<minor version>.<build number>.<4th element>
    '
    ' Return values:
    '  -1 (if appVer1 < appVer2)
    '  0 (if appVer1 = appVer2)
    '  1 (if appVer1 > appVer2)
    '  Null (if appVer1 or appVer2 is Null or appVer1 or appVer2 has invalid format).
    '
    ' Remarks:
    '  - Version <major version>.<minor version>.<build number> will be extended to have <4th element>
    '    by IsValidMSIVersion.
    '  - No error should be raised within the function.
    '  - Return values are based on build in StrComp function.
    '  - http://www.w3schools.com/vbscript/func_strcomp.asp

    Dim versionPattern, ret, strMessage, errSource, ver1, ver2, version, splittedAppVer1, splittedAppVer2

    On Error Resume Next

    errSource = "CommonFunctions::AppVersionComp: "

    ' set default return value
    AppVersionComp = Null

    ' Validate function arguments
    ' Validate appVer1

    ret = IsValidMSIVersion(appVer1, splittedAppVer1)

    If ( (ret <> True) Or (IsNull(ret)) ) Then
        ' check for errors
        strMessage = errSource & "IsValidMSIVersion(appVer1, splittedAppVer1) returned: " & ret & ". " & appVer1 & " is not valid msi version."
        Call WriteToLogFile(strMessage, ERROR_C)
        Err.Clear
        Exit Function
    End If

    ' Validate appVer2
    ret = IsValidMSIVersion(appVer2, splittedAppVer2)

    If ( (ret <> True) Or (IsNull(ret)) ) Then
        ' check for errors
        strMessage = errSource & "IsValidMSIVersion(appVer2, splittedAppVer2) returned: " & ret & ". " & appVer2 & " is not valid msi version."
        Call WriteToLogFile(strMessage, ERROR_C)
        Err.Clear
        Exit Function
    End If

    ' assume that versions are equal
    AppVersionComp = 0

    ' 0 - MAJOR
    ' 1 - MINOR
    ' 2 - REVISION
    ' 3 - 4TH ELEMENT
    For version = 0 To 3
        If (splittedAppVer1(version) > splittedAppVer2(version)) Then
            ' set return value
            AppVersionComp = 1
            Exit Function
        ElseIf (splittedAppVer1(version) < splittedAppVer2(version))Then
            ' set return value
            AppVersionComp = -1
            Exit Function
        End If
    Next

End Function

Function IsValidMSIVersion (ByVal appVer, ByRef splittedAppVer)
    ' Checks whether appVer is valid MSI version (compliant with sys-033)
    ' [appVer]         In - string containing MSI version
    ' [splittedAppVer] In - array, to which IsValidMSIVersion saves three numbers, which assemble MSI version

    ' Return values:
    '   True  - appVer is valid MSI version. No error should be raised.
    '   False - appVer is NOT valid MSI version. No error should be raised.
    '   Null  - Error encountered during function execution. This error should not be propagated.

    ' Remarks:
    '   - No error should be raised within the function.
    '   - aaa.bbb.ccccc Product Version will be appended with the 4th element and be treated as
    '     aaa.bbb.ccccc.ddddd

    ' According to sys-033:
    ' In The Product Version must contain only numbers, and shall be formatted is as follows:
    '   aaa.bbb.ccccc
    ' Where:
    '   aaa    major version number (0  255)
    '   bbb    minor version number (0  255)
    '   ccccc  build number (0  65,535)
    '
    ' Additionally, we have to support 4th number for Product Version, which means that the
    ' following format is supported:
    '   aaa.bbb.ccccc.ddddd
    ' Where:
    '   aaa    major version number (0  255)
    '   bbb    minor version number (0  255)
    '   ccccc  build number (0  65,535)
    '   ddddd  4th number (0  65,535)
    '
    ' e.g. bbb means that there should be 1 to 3 digits.

    Dim versionPattern, errSource, ret, strMessage

    On Error Resume Next

    versionPattern = "^\d{1,3}\.\d{1,3}\.\d{1,5}(\.\d{1,5})?$"
    errSource = "CommonFunctions::IsValidMSIVersion: "
    IsValidMSIVersion = Null

    ret = InStrRegex(appVer, versionPattern)

    ' check for errors
    strMessage = errSource & "Could not check if appVer is valid msi version."
    If (CheckForErrors(strMessage) = True) Then
        ' We return Null if error occured
        IsValidMSIVersion = Null
        Exit Function
    End If

    ' Return false if versionPattern was not found in appVer at position 0
    If (ret <> 0) Then
        IsValidMSIVersion = False
        Exit Function
    End If

    ' split appVer
    splittedAppVer = Split(appVer, ".")

    ' Convert splittedAppVer elements to Long (Integer's range is -32768 and 32767). It is necessary to convert e.g. 003 to 3
    splittedAppVer(0) = CLng(splittedAppVer(0))
    splittedAppVer(1) = CLng(splittedAppVer(1))
    splittedAppVer(2) = CLng(splittedAppVer(2))

    ' Check MSI version length
    If ((UBound(splittedAppVer) + 1) < 4) Then
        ' aaa.bbb.ccccc
        ' extend array size
        Redim Preserve splittedAppVer(UBound(splittedAppVer) + 1)
        ' append array with 0 for the 4th element
        splittedAppVer(UBound(splittedAppVer)) = 0
    Else
        ' aaa.bbb.ccccc.ddddd
        ' convert element to Long (Integer's range is -32768 and 32767)
        splittedAppVer(3) = CLng(splittedAppVer(3))
    End If

    ' re turn False, if any number in appVer is bigger than permissible
    If ( (splittedAppVer(0) > 255) Or _
         (splittedAppVer(1) > 255) Or _
         (splittedAppVer(2) > 65535) Or _
         (splittedAppVer(3) > 65535) ) Then
            IsValidMSIVersion = False
            Exit Function
    End If

    IsValidMSIVersion = True

End Function

'---------------------------------------------------------------------------------------------------------------------------------------------------
' Default implemtation
Function DefaultPreCheck(ByVal appName, ByVal logFileName)

    Dim isInstalled
    Dim strMessage

    ' set default return value
    DefaultPreCheck = SUCCESS

    On Error Resume Next

    ' get installation status
    isInstalled = IsAppInstalled(appName, GetAppBitInfo(appName))

    ' check for errors
    If (IsNull(isInstalled)) Then
        strMessage = "Could not check installation status for application " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        DefaultPreCheck = FAIL
        Exit Function
    End If

    ' check if installed
    If (isInstalled = True) Then
        strMessage = appName & " is already installed."
        Call WriteToLogFile(strMessage, INFO_C)
        If (Err.Number <> 0) Then
            DefaultPreCheck = FAIL
            Err.Clear
            Exit Function
        End If

        ' It is adding all missing registry keys.
        If (FixRegistryKeys(appName, isInstalled) = FAIL) Then
            DefaultPreCheck = FAIL
            Exit Function
        End If

        Dim appVersion, appVerComp
        ' get installed version
        appVersion = GetAppVersion(appName)
        ' check for errors
        If (Not IsEmpty(appVersion)) Then
            strMessage = appName & " version found: " & appVersion
            Call WriteToLogFile(strMessage, INFO_C)
            If (Err.Number <> 0) Then
                DefaultPreCheck = FAIL
                Err.Clear
                Exit Function
            End If

            ' compare installed version with installing version
            appVerComp = AppVersionComp(appVersion, appMsiVer)

            ' perform operations depending on version found
            If Not IsNull(appVerComp) Then
                ' check comparision result
                If (appVerComp = -1) Then
                    ' appVersion < appMsiVer
                ElseIf (appVerComp = 0) Then
                    ' appVersion = appMsiVer
                ElseIf (appVerComp = 1) Then
                    ' appVersion > appMsiVer
                    ' set return value
                    If(Applications.Exists(appName) = True) Then
                        DefaultPreCheck = ERROR_1638
                        strMessage = "Newer version " & appVersion & " is already installed. It is allowed scenario for third party application"
                        Call WriteToLogFile(strMessage, WARNING_C)
                    Else
                    DefaultPreCheck = FAIL
                    strMessage = "Newer version " & appVersion & " is already installed."
                    Call WriteToLogFile(strMessage, ERROR_C)
                    End If
                    Err.Clear
                    Exit Function
                End If
            Else
                ' set return value
                DefaultPreCheck = FAIL
                strMessage = "Could not compare software versions."
                Call WriteToLogFile(strMessage, ERROR_C)

                Err.Clear
                Exit Function
            End If
        Else
            ' set return value
            DefaultPreCheck = FAIL
            strMessage = "Could not check installed version for " & appName & ". Error number is " & Err.Number & ": " & Err.Description
            Call WriteToLogFile(strMessage,ERROR_C)

            Err.Clear
        End If
    End If

End Function

Function DefaultThirdPartyPostInstall (ByVal appName)
    Dim oldVersion
    Dim strMessage
    Dim appVerComp

    On Error Resume Next

    DefaultThirdPartyPostInstall = SUCCESS

    oldVersion = GetAppProperty(appName, CURRENTVERSION)
    ' compare previous version with just installed version
    appVerComp = AppVersionComp(oldVersion, appMsiVer)

    ' perform operations depending on version found
    If Not IsNull(appVerComp) Then
        ' we are only interested in upgrade scenerio. It is checked based on reg keys.
        If (appVerComp = -1) Then
        ' oldVersion < appMsiVer
            If (UpgradeKeys(appName, oldVersion) = FAIL) Then
                DefaultThirdPartyPostInstall = FAIL
                Exit Function
            End If
        End If
    Else
        ' set return value
        DefaultThirdPartyPostInstall = FAIL
        strMessage = "Could not compare software versions."
        Call WriteToLogFile(strMessage, ERROR_C)

        Err.Clear
        Exit Function
    End If

    If(UpdateAppProperty(appName, SACONFIGVER, appConfigVer) = FAIL) Then
        DefaultThirdPartyPostInstall = FAIL
        Exit Function
    End If

    If ( ("Remote Desktop Updates" = appName) Or ("Internet Explorer" = appName) Or ("Microsoft Windows Management Framework" = appName) ) Then
        ' We can exit with SUCCESS for Remote Desktop Updates app, which doe not have a GUID
        DefaultThirdPartyPostInstall = SUCCESS
        Exit Function
    End If

    If(UpgradeGUID(appName) = FAIL) Then
        DefaultThirdPartyPostInstall = FAIL
    End If

End Function

Function DefaultMotorolaPostInstall (ByVal appName)
    DefaultMotorolaPostInstall = SUCCESS

    If(UpdateAppProperty(appName, SACONFIGVER, appConfigVer) = FAIL) Then
        DefaultMotorolaPostInstall = FAIL
        Exit Function
    End If

End Function

Function IsQuickFixInstalled(ByVal fixID)
    ' Checks whether update for Windows (Quick Fix Engineering) is installed
    ' [fixID]         In - HotFixID of update we want to check

    ' Return values:
    '   True  - found. No error should be raised.
    '   False - not found. No error should be raised.
    '   Null  - Error encountered during function execution. This error should not be propagated.

    ' Remarks:
    '  No error should be raised within the function.
    Dim oWmi, objList, strComputer, strMessage, errSource

    On Error Resume Next

    errSource = "CommonFunctions::IsQuickFixInstalled: "

    ' Set default return value
    IsQuickFixInstalled = Null

    If ( IsValidString(fixID) = False ) Then
        Call WriteToLogFile(errSource & "fixID is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    'Parameters are valid here, check if process/service is running
    strComputer = "."
    Set oWmi = GetObject("winmgmts:{impersonationLevel=impersonate}!\\" & strComputer & "\root\cimv2")
    Set objList = oWmi.ExecQuery("SELECT * FROM Win32_QuickFixEngineering WHERE HotFixID = '" & fixID & "'")

    ' Get number of running instances of searched process/service
    If (objList.Count > 0) Then
        IsQuickFixInstalled = True
    Else
        IsQuickFixInstalled = False
    End If

    ' Check for errors. If command called in ExecQuery is wrong, object is returned and no error is raised.
    ' Error is raised upon attempt to make operation on this faulty object (e.g. method .Count)
    strMessage = errSource & "Could not list matching QuickFixEngineering entries. "
    If (CheckForErrors(strMessage) = True) Then
        ' We return Null if error occured
        IsQuickFixInstalled = Null
    End If

End Function

Function IsRemoteDesktopUpdatesInstalled
    ' Checks whether Remote Dektop Updates is installed. Is called in IsAppInstalled.

    ' Return values:
    '   True  - installed. No error should be raised.
    '   False - not installed. No error should be raised.
    '   Null  - Error encountered during function execution. This error should not be propagated.

    ' Remarks:
    '  No error should be raised within the function.
    Dim osVersion, strMessage, errSource, fix1, fix2

    On Error Resume Next

    ' Set default return value
    IsRemoteDesktopUpdatesInstalled = Null

    errSource = "CommonFunctions::IsRemoteDesktopUpdatesInstalled: "

    ' Check OS Version.
    osVersion = GetOSVersion()

    ' Check for errors
    strMessage = errSource & "Could not check operating system version. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' fix1 corresponds to HotFixID 969084
    ' fix2 corresponds to HotFixID 925876
    ' Default value for them is false - not installed
    fix1 = False
    fix2 = False

    If (osVersion = "XP") Then
        fix1 = IsQuickFixInstalled("KB969084")
        fix2 = IsQuickFixInstalled("KB925876")
    ElseIf (osVersion = "2003") Then
        fix2 = IsQuickFixInstalled("KB925876")
    ElseIf (osVersion = "Vista") Then
        fix1 = IsQuickFixInstalled("KB969084")
    Else
        strMessage = "On this Windows version Remote Desktop Updates are not required. They are required on versions: XP, 2003 and Vista."
        Call WriteToLogFile(strMessage, INFO_C)
        If (Err.Number <> 0) Then
            IsRemoteDesktopUpdatesInstalled = Null
            Err.Clear
            Exit Function
        End If
        IsRemoteDesktopUpdatesInstalled = True
        Exit Function
    End If

    If ( IsNull(fix1) Or IsNull(fix2) ) Then
        ' return Null if at least once function IsQuickFixInstalled returned Null - error during execution
        IsRemoteDesktopUpdatesInstalled = Null
    ElseIf ( (fix1 = True) Or (fix2 = True) ) Then
        ' return True if at least one of fixes is found
        IsRemoteDesktopUpdatesInstalled = True
    Else
        IsRemoteDesktopUpdatesInstalled = False
    End If

End Function

Function IsIEReadyToBeInstalled
    ' Checks whether Internet Explorer 11 is ready to be installed. It is called in IsAppInstalled.
    ' State before required reboot.'
    ' Return values:
    '   True  - installed. No error should be raised.
    '   False - not installed. Error should be raised or error encountered during function execution. This error should not be propagated.
    '   Null  - error in IsQuickFixInstalled function

    ' Remarks:
    '  No error should be raised within the function.

    On Error Resume Next

    ' Set return value
    IsIEReadyToBeInstalled = IsQuickFixInstalled("KB2841134")

End Function

Function IsWMFInstalled
    ' Checks whether Windows Management Framework is installed. It is called in IsAppInstalled.
    ' Return values:
    '   True  - installed. No error should be raised.
    '   False - not installed. Error should be raised or error encountered during function execution. This error should not be propagated.
    '   Null  - error in IsQuickFixInstalled function

    ' Remarks:
    '  No error should be raised within the function.

    On Error Resume Next

    Call WriteToLogFile("Check if Microsoft Windows Management Framework 4.0 (KB2819745) is installed" , INFO_C)
    IsWMFInstalled = IsQuickFixInstalled("KB2819745")

End Function

Sub CreateFolderTree(dirPath)
    ' Creates requested directory recursively

    On Error Resume Next
    Dim objFSO
    Set objFSO = CreateObject("Scripting.FileSystemObject")

    If Not objFSO.FolderExists(objFSO.GetParentFolderName(dirPath)) Then
        CreateFolderTree(objFSO.GetParentFolderName(dirPath))
    End If
    If Not objFSO.FolderExists(dirPath) Then
        objFSO.CreateFolder(dirPath)
    End If
End Sub

Function DeleteFolder(ByVal dirPath, ByVal force)
    ' Deletes requested directory recursively
    ' [dirPath] In - directory to delete path
    ' [force]   In - True/False. If True, force deletion - ignore read/only files under dir

    ' Return values:
    '   FAIL  - failed to delete directory
    '   SUCCESS - directory deleted or not exist

    On Error Resume Next
    Dim objFSO, strMessage, errSource
    Set objFSO = CreateObject("Scripting.FileSystemObject")

    errSource = "CommonFunctions::DeleteFolder: "
    DeleteFolder = FAIL

    ' Validate arguments
    If IsValidString(dirPath) = False Then
        strMessage = errSource & "Variable dirPath is not initialized. Application name: " & appName
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If


    If objFSO.FolderExists(dirPath) Then
        objFSO.deleteFolder dirPath, force

        ' Check for errors
        strMessage = errSource & "Failed to delete folder " & dirPath & ". Application name: " & appName
        If(CheckForErrors(strMessage) = True) Then
            Exit Function
        End If
    End If

    ' Check for errors
    strMessage = errSource & "Failed to check if folder " & dirPath & " exists. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    DeleteFolder = SUCCESS
End Function


Function RemoveFileAttribute(ByVal fileName, ByRef attr)
    ' Removes specific attribute(s) from file
    '
    ' [fileName] In - File name with full file path
    ' [attr]     In - Attribute to remove. Can be value or array
    '
    ' Return value:
    '  SUCCESS - attribute(s) removed or has not been previously set
    '  FAIL - unable to remove attribute from file
    '
    ' Remarks:
    '  If return value is FAIL, it is recommended to check for errors to make sure that function has been properly executed.

    Dim objFSO, currentFile, iter, strMessage, errSource

    On Error Resume Next

    ' Set default return value
    RemoveFileAttribute = FAIL

    errSource = "CommonFunctions::RemoveFileAttribute: "

    If Not IsValidString(fileName) Then
        strMessage = errSource & "Filename is Null or Empty. Application name: " & appName
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    Set objFSO = CreateObject("Scripting.FileSystemObject")
    Set currentFile = objFSO.GetFile(fileName)

    ' Check for errors
    strMessage = errSource & "Failed to create Scripting.FileSystemObject. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If (IsArray(attr) = True) Then

        For iter = 0 To UBound(attr)
            If Not IsNumeric(attr(iter)) Then
                strMessage = errSource & "At least one entry in 'attr' array is not numeric." & " Application name: " & appName
                Call WriteToLogFile(strMessage, ERROR_C)
                Exit Function
            End If

            If(currentFile.Attributes And attr(iter)) Then
                currentFile.Attributes = currentFile.Attributes XOR attr(iter)
            End If

            If(currentFile.Attributes And attr(iter)) Then
                strMessage = errSource & "Cannot remove " & attr(iter) & " attribute from " & fileName & ". " & " Application name: " & appName
                Call WriteToLogFile(strMessage, ERROR_C)
                Exit Function
            End If

        Next

    Else
        If Not IsNumeric(attr) Then
            strMessage = errSource & "None numeric value passed as 'attr' parameter." & " Application name: " & appName
            Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If

        If(currentFile.Attributes And attr) Then
            currentFile.Attributes = currentFile.Attributes XOR attr
        End If

        If(currentFile.Attributes And attr) Then
            strMessage = errSource & "Cannot remove " & attr & " attribute from " & fileName & ". " & " Application name: " & appName
            Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If

    End If

    RemoveFileAttribute = SUCCESS

End Function

Function SetJavaDeploymentConfigProperty(propertyName, value, javaAppName)
    ' Sets value of property in java system deployment config file (C:\Windows\Sun\Java\Deployment\deployment.config)

    ' Input values:
    '   propertyName  - property name e.g. deployment.security.revocation.check
    '   value         - value to set for propertyName
    '
    ' Return values:
    '   SUCCESS or FAIL

    On Error Resume Next

    Dim objFSO, objSystemProperties, strLine, errSource, strMessage
    Dim patternToSearch, linesToRetain, lineToAdd

    errSource = "CommonFunctions::SetJavaDeploymentConfigProperty: "
    SetJavaDeploymentConfigProperty = FAIL

    'Validate arguments
    If ( IsValidString(propertyName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'propertyName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( IsValidString(value) = False ) Then
        Call WriteToLogFile(errSource & "argument 'value' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    lineToAdd = propertyName & "=" & value

    ' replace all '.' with '\.' - string will be used for regexp
    patternToSearch = Replace(propertyName,Chr(46),Chr(92) & Chr(46))

    ' Check for errors
    strMessage = errSource & "Replace function failed. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' All file content not related to property we want to set. We need to retain it.
    linesToRetain = ""

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    ' Check for errors
    strMessage = errSource & "Failed to create Scripting.FileSystemObject. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Dim javaSystemDeploymentConfig
    javaSystemDeploymentConfig = GetJavaAppFolderPath(javaAppName) & "lib\deployment.config"
    If(IsNull(javaSystemDeploymentConfig)) Then
        Exit Function
    End If

    If (objFSO.FileExists(javaSystemDeploymentConfig)) Then

        ' Ensure that deployment.config file is "writable"
        Set objSystemProperties = objFSO.GetFile(javaSystemDeploymentConfig)

        If (SUCCESS <> RemoveFileAttribute(javaSystemDeploymentConfig, Array(READ_ONLY, HIDDEN))) Then
            Call WriteToLogFile("Failed to change file attribute(s)", ERROR_C)
        Exit Function
        End If

        Set objSystemProperties = objFSO.OpenTextFile(javaSystemDeploymentConfig, FOR_READING, False)

        ' Check for errors
        strMessage = errSource & "Failed to open deployment.config in system location (" & javaSystemDeploymentConfig & ") for reading. Application name: " & appName
        If(CheckForErrors(strMessage) = True) Then
            Exit Function
        End If

        Do Until objSystemProperties.AtEndOfStream
            strLine = objSystemProperties.ReadLine

            If (InStrRegex(strLine, patternToSearch) < 0) Then
                linesToRetain = linesToRetain & strLine & vbCrLf
            End If

            ' Check for errors
            strMessage = errSource & "Error thrown while searching for pattern " & patternToSearch & " in line " & strLine & " from deployment.config in system location (" & javaSystemDeploymentConfig & "). Application name: " & appName
            If(CheckForErrors(strMessage) = True) Then
                Exit Function
            End If
        Loop
        objSystemProperties.Close
    End If

    ' Check for errors
    strMessage = errSource & "Failed to check if deployment.config in system location (" & javaSystemDeploymentConfig & ") exists. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Set objSystemProperties = objFSO.OpenTextFile(javaSystemDeploymentConfig, FOR_WRITING, True)

    ' Check for errors
    strMessage = errSource & "Failed to open deployment.config in system location (" & javaSystemDeploymentConfig & ") for writing. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objSystemProperties.Write(linesToRetain)
    objSystemProperties.WriteLine(lineToAdd)

    ' Check for errors
    strMessage = errSource & "Failed to write content to deployment.config in system location (" & javaSystemDeploymentConfig & "). Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objSystemProperties.Close

    SetJavaDeploymentConfigProperty = SUCCESS
End Function

Function SetJavaDeploymentProperty(propertyName, value, javaAppName)
    ' Sets value of property in java system deployment properties file (C:\Windows\Sun\Java\Deployment\deployment.properties)

    ' Input values:
    '   propertyName  - property name e.g. deployment.security.revocation.check
    '   value         - value to set for propertyName
    '
    ' Return values:
    '   SUCCESS or FAIL

    On Error Resume Next

    Dim objFSO, objSystemProperties, strLine, errSource, strMessage
    Dim patternToSearch, linesToRetain, lineToAdd

    errSource = "CommonFunctions::SetJavaDeploymentProperty: "
    SetJavaDeploymentProperty = FAIL

    'Validate arguments
    If ( IsValidString(propertyName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'propertyName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( IsValidString(value) = False ) Then
        Call WriteToLogFile(errSource & "argument 'value' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    ' if value is EMPTY_VALUE, don't write it down
    If ( value = "EMPTY_VALUE" ) Then
            lineToAdd = propertyName
        Else
            lineToAdd = propertyName & "=" & value
    End If

    ' replace all '.' with '\.' - string will be used for regexp
    patternToSearch = Replace(propertyName,Chr(46),Chr(92) & Chr(46))

    ' Check for errors
    strMessage = errSource & "Replace function failed. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' All file content not related to property we want to set. We need to retain it.
    linesToRetain = ""

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    ' Check for errors
    strMessage = errSource & "Failed to create Scripting.FileSystemObject. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Dim javaSystemDeploymentProperties
    javaSystemDeploymentProperties = GetJavaAppFolderPath(javaAppName) & "lib\deployment.properties"
    If(IsNull(javaSystemDeploymentProperties)) Then
        Exit Function
    End If

    If(objFSO.FileExists(javaSystemDeploymentProperties)) Then

        ' Ensure that deployment.properties file is "writable"
        Set objSystemProperties = objFSO.GetFile(javaSystemDeploymentProperties)

        If (SUCCESS <> RemoveFileAttribute(javaSystemDeploymentProperties, Array(READ_ONLY, HIDDEN))) Then
            Call WriteToLogFile("Failed to change file attribute(s)", ERROR_C)
        Exit Function
        End If

        Set objSystemProperties = objFSO.OpenTextFile(javaSystemDeploymentProperties, FOR_READING, False)

        ' Check for errors
        strMessage = errSource & "Failed to open deployment.properties in system location (" & javaSystemDeploymentProperties & ") for reading. Application name: " & appName
        If(CheckForErrors(strMessage) = True) Then
            Exit Function
        End If

        Do Until objSystemProperties.AtEndOfStream
            strLine = objSystemProperties.ReadLine

            If (InStrRegex(strLine, patternToSearch) < 0) Then
                linesToRetain = linesToRetain & strLine & vbCrLf
            End If

            ' Check for errors
            strMessage = errSource & "Error thrown while searching for pattern " & patternToSearch & " in line " & strLine & " from deployment.properties in system location (" & javaSystemDeploymentProperties & "). Application name: " & appName
            If(CheckForErrors(strMessage) = True) Then
                Exit Function
            End If
        Loop
        objSystemProperties.Close
    End If

    ' Check for errors
    strMessage = errSource & "Failed to check if deployment.properties in system location (" & javaSystemDeploymentProperties & ") exists. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Set objSystemProperties = objFSO.OpenTextFile(javaSystemDeploymentProperties, FOR_WRITING, True)

    ' Check for errors
    strMessage = errSource & "Failed to open deployment.properties in system location (" & javaSystemDeploymentProperties & ") for writing. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objSystemProperties.Write(linesToRetain)
    objSystemProperties.WriteLine(lineToAdd)

    ' Check for errors
    strMessage = errSource & "Failed to write content to  deployment.properties in system location (" & javaSystemDeploymentProperties & "). Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objSystemProperties.Close

    SetJavaDeploymentProperty = SUCCESS
End Function

Function SetJavaConfigProperties(pathToJavaFolder)
    ' Handles creation of java system deployment config file:
    '   - "...\lib\deployment.config"
    On Error Resume Next

    Dim objFSO, objFileConfig
    Dim strDeploymentConfigFullPath, strLine
    Dim strMessage, errSource

    SetJavaConfigProperties = FAIL
    errSource = "CommonFunctions::SetJavaConfigProperties: "

    If Not IsValidString(pathToJavaFolder) Then
        Exit Function
    End If

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " &appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Create deployment.config file in Java \lib directory
    ' which contains configuration for Java Control Panel
    strDeploymentConfigFullPath = pathToJavaFolder & "\lib\deployment.config"

    ' Ensure that deployment.config file is "writable"
    If( objFSO.FileExists(strDeploymentConfigFullPath) ) Then
        If (SUCCESS <> RemoveFileAttribute(strDeploymentConfigFullPath, Array(READ_ONLY, HIDDEN))) Then
            Call WriteToLogFile("Failed to change file attribute(s)", ERROR_C)
            Exit Function
        End If
    End If

    Set objFileConfig = objFSO.CreateTextFile(strDeploymentConfigFullPath, True)

    objFileConfig.WriteLine "deployment.system.config.mandatory=True"
    strLine = Replace(pathToJavaFolder, "C:", "")
    strLine = "deployment.system.config=file\:C\:" & Replace(strLine, "\", "/") & "lib/deployment.properties"
    objFileConfig.WriteLine strLine

    strMessage = errSource & "Failed to create file " & strDeploymentConfigFullPath & ". Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objFileConfig.Close
    SetJavaConfigProperties = SUCCESS

End Function

Function SetJavaDeploymentProperties(pathToJavaFolder)
    ' Handles creation and upgrade of java system deployment properties file:
    '   - "...\lib\deployment.properties"

    On Error Resume Next

    'Disable certification revocation list checking
    Dim appDataDeploymentProperties, appDataJavaDeployment
    Dim objFSO, objFileConfig, objSystemProperties, objAppDataProperties, osVersion
    Dim StopTimeout, javaControlPanelPath
    Dim strLine
    Dim ret, strMessage, WshShell, errSource

    SetJavaDeploymentProperties = FAIL

    errSource = "CommonFunctions::SetJavaDeploymentProperties: "

    appDataJavaDeployment = GetAppDataJavaDeploymentPath()

    If Not IsValidString(appDataJavaDeployment) Then
        Exit Function
    End If

    appDataDeploymentProperties = appDataJavaDeployment & "\deployment.properties"

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " &appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Now it is needed to create deployment.properties file for all Javas in ...\lib\ folders.
    ' We want to have almost all default deployment.properties and add only line responsible for disabling CRL check.
    ' Unfortunately, after java installation there is no default deployment.properties on disk.
    ' Default deployment.properties is created in AppData location after running Java Control Panel, so we need to run Java Control Panel (javacpl.exe)
    ' and copy default settings, which matches "deployment\.javaws\.jre\.", to ...\lib\ directory.

    ' Delete deployment.properties file in AppData location, if already exists - we want to be sure that later launched Java Control Panel will create new one.
    If (objFSO.FileExists(appDataDeploymentProperties)) Then
        objFSO.DeleteFile(appDataDeploymentProperties)

        strMessage = errSource & "Failed to delete deployment.properties file in AppData location - " & appDataDeploymentProperties & " . Application name: " & appName & "."
        If(CheckForErrors(strMessage) = True) Then
            Exit Function
        End If
    End If

    strMessage = errSource & "Failed to check if deployment.properties file in AppData location exists (" & appDataDeploymentProperties & "). Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Check where latest java is installed on computer (check in Java registers).
    ' Then set path to java control panel

    Dim currJavaReg, currJavaVersion, currJavaHome

    javaControlPanelPath = pathToJavaFolder & "\bin\javacpl.exe"

    If Not ( objFSO.FileExists(javaControlPanelPath) ) Then
        Call WriteToLogFile("Java Control Panel does not found - file " & javaControlPanelPath & " does not exist. Application name: " & appName & ".", ERROR_C)
        Exit Function
    End If

    ' Make sure that java does not store any information about Single Instance Services - delete them.
    ' If java Single Instance Service was earlier not closed properly, its record can be still
    ' held and prevent from running another instance.
    ret = DeleteJavaDeploymentSI()
    If (FAIL = ret) Then
        Exit Function
    End If

    strMessage = errSource & "Failed to delete java Single Instance Services temporary folder. Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Run Java Control Panel
    Set WshShell = CreateObject("Wscript.Shell")
    WshShell.Exec(javaControlPanelPath)

    strMessage = errSource & "Failed to launch Java Control Panel. Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    'Wait until deployment.properties appears (it appears in user's appData folder)
    Dim startTime
    startTime = Timer()
    Do While Not (objFSO.FileExists(appDataDeploymentProperties))
        'Check at most 600 seconds to appear deployment.properties in AppData location
        If ((Timer() - startTime) > 600) Then
            Call WriteToLogFile("Cannot find deployment.properties in AppData location. Application name: " & appName & ".", ERROR_C)

            ' close Java Control Panel
            Call CloseJavaCPL()
            Exit Function
        Else
            ' appDataDeploymentProperties does not exist, wait additional 100 miliseconds
            WScript.Sleep(100)
        End If
    Loop

    strMessage = errSource & "Failed to check if after running Java Control Panel deployment.properties file in AppData location exists (" & appDataDeploymentProperties & "). Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' We detected that Java Control Panel created appDataDeploymentProperties.
    ' Wait additional 3 seconds so as to be certain that Java Control Panel has time to fill in this file.
    WScript.Sleep(3000)

    ' close Java Control Panel
    If (FAIL = CloseJavaCPL()) Then
        Exit Function
    End If

    Dim patternJreDeployment, patternJreToSkip, linesToRetain, linesToAdd
    Dim javaLibDeploymentProperties
    patternJreDeployment = "deployment\.javaws\.jre\."
    patternJreToSkip = "deployment\.user\.cachedir"
    linesToRetain = ""
    linesToAdd = ""

    javaLibDeploymentProperties = pathToJavaFolder & "\lib\deployment.properties"
    If(objFSO.FileExists(javaLibDeploymentProperties)) Then

        ' Ensure that deployment.properties file is "writable"
        If (SUCCESS <> RemoveFileAttribute(javaLibDeploymentProperties, Array(READ_ONLY, HIDDEN))) Then
            Call WriteToLogFile("Failed to change file attribute(s)", ERROR_C)
        Exit Function
        End If

        Set objSystemProperties = objFSO.OpenTextFile(javaLibDeploymentProperties, FOR_READING, False)

        strMessage = errSource & "Failed to open deployment.properties file in system location (" & javaLibDeploymentProperties & ") for reading. Application name: " & appName & "."
        If(CheckForErrors(strMessage) = True) Then
            Exit Function
        End If

        Do Until objSystemProperties.AtEndOfStream
            strLine = objSystemProperties.ReadLine
            If IsValidString(strLine) And (InStrRegex(strLine, patternJreDeployment) < 0) And (InStrRegex(strLine, patternJreToSkip) < 0) Then
                linesToRetain = linesToRetain & strLine & vbCrLf
            End If

            ' Check for errors
            strMessage = errSource & "Error thrown while searching for pattern " & patternJreDeployment & " in line " & strLine & " from deployment.properties in system location (" & javaLibDeploymentProperties & "). Application name: " & appName
            If(CheckForErrors(strMessage) = True) Then
                Exit Function
            End If
        Loop
        objSystemProperties.Close
    End If

       strMessage = errSource & "Failed to check if after running Java Control Panel deployment.properties file in system location exists (" & appDataDeploymentProperties & "). Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Set objAppDataProperties = objFSO.OpenTextFile(appDataDeploymentProperties, FOR_READING, False)

    strMessage = errSource & "Failed to open deployment.properties file in AppData location (" & appDataDeploymentProperties & ") for reading. Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Do Until objAppDataProperties.AtEndOfStream
        strLine = objAppDataProperties.ReadLine
        If (InStrRegex(strLine, patternJreDeployment) >= 0) Then
            linesToAdd = linesToAdd & strLine & vbCrLf
        End If

        ' Check for errors
        strMessage = errSource & "Error thrown while searching for pattern " & patternJreDeployment & " in line " & strLine & " from deployment.properties in AppData location (" & appDataDeploymentProperties & "). Application name: " & appName
        If(CheckForErrors(strMessage) = True) Then
            Exit Function
        End If
    Loop
    objAppDataProperties.Close

    Set objSystemProperties = objFSO.OpenTextFile(javaLibDeploymentProperties, FOR_WRITING, True)

    strMessage = errSource & "Failed to open file " & javaLibDeploymentProperties & " for writing. Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objSystemProperties.Write(linesToAdd)
    objSystemProperties.Write(linesToRetain)

    strMessage = errSource & "Failed to write defualt content to java system deployment.properties (" & javaLibDeploymentProperties & "). Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objSystemProperties.Close

    SetJavaDeploymentProperties = SUCCESS

End Function

Function CopyDeploymentPropertiesFile(appName)
    ' Copies deployment.properties file to another java 32 ...\lib\ location
    '  In: current application name

    ' Return value:
    '  SUCCESS - file copied or no other java installed
    '  FAIL - error during process

    On Error Resume Next

    Dim SearchedApp, strPathDest, strPathSource, isInstalled, strMessage
    Dim Java7Name, Java8Name
    Java7Name = "Motorola ASTRO Java 7 Static"
    Java8Name = "Motorola ASTRO Java Family"

    CopyDeploymentPropertiesFile = FAIL

    If(appName = Java8Name) Then
        SearchedApp = Java7Name
    Else
        SearchedApp = Java8Name
    End If

    isInstalled = IsAppInstalled(SearchedApp, GetAppBitInfo(appName))
    If (IsNull(isInstalled)) Then
        strMessage = "Could not check installation status for application " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    If(isInstalled = true) Then

        strPathDest = GetJavaAppFolderPath(SearchedApp)
        If(IsNull(strPathDest)) Then
            Exit Function
        End If
        strPathSource = GetJavaAppFolderPath(appName)
        If(IsNull(strPathSource)) Then
            Exit Function
        End If

        If(CopyFileToFolder((strPathSource & "lib\deployment.properties"), (strPathDest & "lib")) = FAIL) Then
            Exit Function
        End If

    End If

    if (IsAppInstalled(Java7Name, GetAppBitInfo(appName))) Then
        Dim strCacheFolder, oShell, retCode
        Set oShell = CreateObject("WScript.Shell")
        strCacheFolder = "..\\AppData\\LocalLow\\Sun\\Java\\Deployment\\cache7"
        If (setJavaDeploymentProperty("deployment.user.cachedir", strCacheFolder, Java7Name) = FAIL) Then
            strMessage = "Could not update deployment.properties for " & Java7Name & "."
            Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If
    End If

    CopyDeploymentPropertiesFile = SUCCESS

End Function

Function GetJavaAppFolderPath(appName)
    ' Returnes path to Java folder application
    '  In: current application name

    ' Return value:
    '  Path - path to Java folder application
    '  Null - in case of error

    On Error Resume Next

    GetJavaAppFolderPath = Null

    Dim strUninstallPath, strGuid, strRegisterPath, strMessage, strPath

    strUninstallPath = GetUninstallPath(GetAppBitInfo(appName))
    strGuid          = GetAppGUID(appName, GetAppBitInfo(appName))
    strRegisterPath  = "HKLM\" & strUninstallPath & "\" & strGuid & "\InstallLocation"
    strPath          = GetProperty(strRegisterPath)

    If(IsNull(strPath)) Then
        strMessage = "Failed to get folder path for application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
    End If

    GetJavaAppFolderPath = strPath

End Function

Function ConfigJavaDeploymentProperties()
    ' Handles creation of java deployment.config and deploymen.properties files
    '   - both should be created and put to java home\lib\ folder

    On Error Resume Next

    ConfigJavaDeploymentProperties = FAIL

    Dim strAppNamePath
    strAppNamePath = GetJavaAppFolderPath(appName)
    If(IsNull(strAppNamePath)) Then
        Exit Function
    End If

    If(SetJavaConfigProperties(strAppNamePath) <> SUCCESS) Then
        Exit Function
    End If

    If(SetJavaDeploymentProperties(strAppNamePath) <> SUCCESS) Then
        Exit Function
    End If

    ConfigJavaDeploymentProperties = SUCCESS

End Function

Function GetAppDataJavaDeploymentPath
    ' Returns path to java deployment folder for current user

    ' Return values:
    '   Null  - failed to determine path
    '   Other - path to java deployment folder

    On Error Resume Next

    Dim strMessage, WshShell, errSource, osVersion, userprofile

    GetAppDataJavaDeploymentPath = Null

    errSource = "CommonFunctions::GetAppDataJavaDeploymentPath: "

    Set WshShell = CreateObject("Wscript.Shell")
    userprofile = wshShell.ExpandEnvironmentStrings("%USERPROFILE%")

    ' Check OS Version.
    osVersion = GetOSVersion()

    ' Check for errors
    strMessage = errSource & "Could not check operating system version. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Determine where to search for deployment.properties file in AppData location, depending on operating system
    If (osVersion = "XP") Then
        GetAppDataJavaDeploymentPath = userprofile & "\Application Data\Sun\Java\Deployment"
    Else
        GetAppDataJavaDeploymentPath = userprofile & "\AppData\LocalLow\Sun\Java\Deployment"
    End If
End Function

Function DeleteJavaDeploymentSI
    ' Deletes folder ...\AppData\LocalLow\Sun\Java\Deployment\tmp\si, which
    ' stores informations about java Single Instance Services - e.g. Java Control Panel.
    ' This function should be used after killing all java instances, to make sure that
    ' Java Runtime Environment does not consider any of Single Instance Service as running

    ' Return values:
    '   Null  - failed to determine path
    '   Other - path to java deployment folder

    On Error Resume Next

    Dim strMessage, errSource, appDataJavaDeployment, javaDeploymentSI, ret

    DeleteJavaDeploymentSI = FAIL

    errSource = "CommonFunctions::DeleteJavaDeploymentSI: "

    ' Remove temporary files from deployment
    appDataJavaDeployment = GetAppDataJavaDeploymentPath()

    If Not IsValidString(appDataJavaDeployment) Then
        Exit Function
    End If

    javaDeploymentSI = appDataJavaDeployment & "\tmp\si"

    ' Delete ...\AppData\LocalLow\Sun\Java\Deployment\tmp\si folder recusively and forcefully
    ret = DeleteFolder(javaDeploymentSI, True)
    If (FAIL = ret) Then
        Exit Function
    End If

    DeleteJavaDeploymentSI = SUCCESS
End Function

Function CloseJavaCPL
    ' Closes Java Control Panel. At first, sends termination signal,
    ' if Java Control Panel was not closed, function returns FAIL
    ' and before exiting closes Java Control Panel forcefully and
    ' remove java Single Instance Services temporary folder

    ' Return values:
    '   SUCCESS - termination signal made Java Control Panel close
    '   FAIL    - Java Control Panel was not closed after termination signal

    On Error Resume Next

    Dim WshShell, strMessage, errSource, cmdCloseJavaCPL, startTime

    CloseJavaCPL = FAIL

    errSource = "CommonFunctions::CloseJavaCPL: "

    ' Command to close Java Control Panel
    cmdCloseJavaCPL = TASKKILL & " /FI " & Chr(34) & "WindowTitle eq Java Control Panel" & Chr(34) & ""

    Set WshShell = CreateObject("Wscript.Shell")

    ' Check if Java Control Panel was closed
    startTime = Timer()
    Do While (TRUE = isMotorolaJavaRunning())
        If ((Timer() - startTime) > (DEFAULT_STOP_TIMEOUT / 1000)) Then
            strMessage = errSource & "Failed to softly stop javaw.exe (underlying Java Control Panel) is running. Application name: " & appName & "."
            Call WriteToLogFile(strMessage, WARNING_C)

            ' Close Java Control Panel forcefully
            If (FAIL = StopMotorolaJavasByPath()) Then
                strMessage = errSource & "Failed to forcefully stop javaw.exe (underlying Java Control Panel) is running. Application name: " & appName & "."
                Call WriteToLogFile(strMessage, ERROR_C)
                Exit Function
            End If

            ' Delete java Single Instance Services temporary files - they could harm future java installation
            If (FAIL = DeleteJavaDeploymentSI()) Then
                strMessage = errSource & "Failed to delete java Single Instance Services tmp folder. Application name: " & appName & "."
                Call WriteToLogFile(strMessage, WARNING_C)
            End If
        Else
            ' Send termination signal to Java Control Panel
            WshShell.Run cmdCloseJavaCPL

            strMessage = errSource & "Failed to run command: " & Chr(34) & cmdCloseJavaCPL & Chr(34) & ". Application name: " & appName & "."
            If(CheckForErrors(strMessage) = True) Then
                Exit Function
            End If

            WScript.Sleep(1000)
        End If
    Loop

    strMessage = errSource & "Failed to check if process javaw.exe (underlying Java Control Panel) is running. Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    CloseJavaCPL = SUCCESS
End Function

Function CheckIfDriveExists(ByVal driveLetter, ByVal driveType)
    ' Function to check if given drive exists
    ' [driveLetter]    Drive letter (example: 'C:')
    ' [driveType]      Drive type
    '                  - DRIVE_REMOVABLE
    '                  - DRIVE_FIXED
    '                  - DRIVE_NETWORK
    '                  - DRIVE_CDROM
    '                  - DRIVE_RAMDISK
    '
    ' Return values:
    '   True  - Given drive exists
    '   False - Given drive does not exist
    '   Null  - Error encountered during function execution. This error should not be propagated

    On Error Resume Next

    Dim objFSO, strMessage, errSource, objDrive

    CheckIfDriveExists = Null
    errSource = "CommonFunctions::CheckIfDriveExists: "

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    'Validate arguments
    If ( IsValidString(driveLetter) = False ) Then
        Call WriteToLogFile(errSource & "argument 'driveLetter' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( IsValidString(driveType) = False ) Then
        Call WriteToLogFile(errSource & "argument 'driveType' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If objFSO.DriveExists(driveLetter) Then
        If ( objFSO.GetDrive(driveLetter).DriveType = driveType ) Then
            CheckIfDriveExists = True
        Else
            CheckIfDriveExists = False
        End If

        strMessage = "Error occured during checking drive type"
        If(CheckForErrors(strMessage) = True) Then
            Exit Function
        End If
    Else
        CheckIfDriveExists = False
        strMessage = errSource & "Drive " & driveLetter & " does not exist. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, WARNING_C)
    End If
End Function

Function CreateUpgradeMetaDataFile(ByVal metaDataFilePath)
    ' Function to create metadata file for upgrade
    ' Metadata file contains basic info about pre upgraded Operating System

    ' Return values:
    '   SUCCESS
    '   FAIL
    On Error Resume Next

    Dim objFSO, objMetaFile, metaDataFile, errSource, strMessage
    Dim osVersion, bitInfo, appVersion

    CreateUpgradeMetaDataFile = FAIL
    errSource = "CommonFunctions::CreateUpgradeMetaDataFile: "

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    If Not objFSO.FolderExists(metaDataFilePath) Then
        strMessage = errSource & "Can not create upgrade metadata file. Path " & metaDataFilePath & " does not exist. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Err.Clear
        Exit Function
    End If
    metaDataFile = metaDataFilePath & "\" & UPGRADE_DATA_STORE_FILE

    osVersion = GetOSVersionNumber()
    strMessage = errSource & "Could not check operating system version. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    bitInfo = GetOSBitInfo()
    strMessage = errSource & "Could not check operating system architecture. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    appVersion = GetAppVersion(appName)
    strMessage = errSource & "Could not check application version. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If


    Set objMetaFile = objFSO.CreateTextFile(metaDataFile, True)

    strMessage = errSource & "Could not create file " & metaDataFile & ". Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objMetaFile.WriteLine("OSVersionNumber" & UPGRADE_DATA_STORE_SEPARATOR & osVersion)
    objMetaFile.WriteLine("OSBitInfo" & UPGRADE_DATA_STORE_SEPARATOR & bitInfo)
    objMetaFile.WriteLine("AppVersion" & UPGRADE_DATA_STORE_SEPARATOR & appVersion)

    objMetaFile.Close

    strMessage = errSource & "Could not write to file " & metaDataFile & ". Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        CreateUpgradeMetaDataFile = FAIL
    Else
        CreateUpgradeMetaDataFile = SUCCESS
    End If
End Function

Function GetPropertiesFromFile(ByVal file, ByVal separator)
    ' Function to read property file and returns dictionary object
    ' with key-value content. Broken lines are omitted and no error is raised.

    ' Return values:
    '   Scripting.Dictionary - on success
    '   Null - on error

    On Error Resume Next

    Dim objFSO, errSource, properties, propertiesFile, arr, strMessage, line
    GetPropertiesFromFile = Null

    errSource = "CommonFunctions::GetPropertiesFromFile: "

    Set properties = CreateObject("Scripting.Dictionary")
    Set objFSO = CreateObject("Scripting.FileSystemObject")

    Set propertiesFile = objFSO.OpenTextFile(file, FOR_READING)
    strMessage = errSource & "Failed to open " & file & " file. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Do While propertiesFile.AtEndOfStream <> True
        line = propertiesFile.ReadLine
        arr = Split(line, separator)
        If (uBound(arr) = 1) Then
            strMessage = errSource & "Loaded: " & line & ". Application name: " & appName & "."
            Call WriteToLogFile(strMessage, INFO_C)
            properties(arr(0)) = arr(1)
        Else
            strMessage = errSource & "Skipping line: " & line & ". Application name: " & appName & "."
            Call WriteToLogFile(strMessage, WARNING_C)
        End If
    Loop

    propertiesFile.Close
    strMessage = errSource & "Could not read file " & file & ". Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    Else
        Set GetPropertiesFromFile = properties
    End If

End Function

Function PrepareUpgradeDataStore()
    ' Function to prepare Upgrade Datastore and its content:
    ' - checks if application is installed
    ' - checks if partition for data retention exists
    ' - ensures and creates datastore folder
    ' - cleans up application folder if already exists
    ' - creates metadata file

    ' Return values:
    '   PREPAREUPGRADE_SUCCESS - partition for data retention exists and all above operations were successfully executed
    '   PREPAREUPGRADE_SKIP - partition for data retention does not exist
    '   PREPAREUPGRADE_FAIL - on error


    On Error Resume Next

    Dim objFSO, ret, appUpgradeFolder, errSource, strMessage

    PrepareUpgradeDataStore = PREUPGRADE_FAIL
    errSource = "CommonFunctions::PrepareUpgradeDataStore: "

    appUpgradeFolder = UPGRADE_DATA_STORE_FOLDER & "\" & appName

    ' get installation status
    ret = IsAppInstalled(appName, GetAppBitInfo(appName))

    ' check for errors
    If (IsNull(ret)) Then
        strMessage = "Could not check installation status for application " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( ret = False ) Then
        strMessage = "Application " & appName & " is not installed. DataStore for data retention won't be prepared."
        Call WriteToLogFile(strMessage, INFO_C)
        PrepareUpgradeDataStore = PREUPGRADE_SKIP
        Exit Function
    End If

    ret = CheckIfDriveExists(UPGRADE_DATA_STORE_DRIVE, DRIVE_FIXED)

    ' check for errors
    If (IsNull(ret)) Then
        strMessage = "Could not check if drive " & UPGRADE_DATA_STORE_DRIVE & " exists. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( ret = False ) Then
        ' check for errors
        strMessage = "Drive " & UPGRADE_DATA_STORE_DRIVE & " does not exist. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Err.Clear
        PrepareUpgradeDataStore = PREUPGRADE_SKIP
        Exit Function
    End If

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If Not objFSO.FolderExists(UPGRADE_DATA_STORE_FOLDER) Then
        objFSO.CreateFolder(UPGRADE_DATA_STORE_FOLDER)
    End If

    strMessage = errSource & "Failed to create " & UPGRADE_DATA_STORE_FOLDER & " folder. Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If objFSO.FolderExists(appUpgradeFolder) Then
        ' Folder already exists, removing content
        If (FAIL = DeleteFolder(appUpgradeFolder, True)) Then
            strMessage = errSource & "DeleteFolder(" & appUpgradeFolder & ") returned: " & ret
            Call WriteToLogFile(strMessage, ERROR_C)
            Err.Clear
            Exit Function
        End If
    End If

    objFSO.CreateFolder(appUpgradeFolder)
    strMessage = errSource & "Failed to create application " & appUpgradeFolder & " folder. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    if (SUCCESS = CreateUpgradeMetaDataFile(appUpgradeFolder)) Then
        PrepareUpgradeDataStore = PREUPGRADE_SUCCESS
    Else
        strMessage = errSource & "Failed to create upgrade datastore metadata file. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
    End If

End Function


Function CopyFileToFolder(ByVal sourceFileName, ByVal destinationFolderName)
    ' Function to copy file to specified folder

    ' [sourceFileName]        In - string, source file name with full path
    ' [destinationFolderName] In - string, destination folder name
    '
    ' Return values:
    '   SUCCESS - when file is copied successfully
    '   FAIL - on error

    On Error Resume Next

    Dim errSource, strMessage, objFSO, fileName, destinationFileName


    errSource = "CommonFunctions::CopyFileToFolder "
    CopyFileToFolder = FAIL

    'Validate arguments
    If ( IsValidString(sourceFileName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'sourceFileName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( IsValidString(destinationFolderName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'destinationFolderName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If


    Set objFSO = CreateObject("Scripting.FileSystemObject")

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Check if source file exists
    If Not objFSO.FileExists(sourceFileName) Then
        strMessage = errSource & "File " & sourceFileName & " does not exist. Cannot continue copy operation. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    ' Get file name without full path
    fileName = objFSO.GetFileName(sourceFileName)

    destinationFileName = objFSO.BuildPath(destinationFolderName, fileName)

    'Check to see if the file already exists in the destination folder
    If objFSO.FileExists(destinationFileName) Then
        'Check to see if the file is read-only
        If Not objFSO.GetFile(destinationFileName).Attributes And READ_ONLY Then
            strMessage = errSource & "File " & destinationFileName & " already exists and will be overwritten. Application name: " & appName & "."
            Call WriteToLogFile(strMessage, WARNING_C)
        Else
            strMessage = errSource & "File " & destinationFileName & " already exists and has ReadOnly attribute. Cannot continue copy operation. Application name: " & appName & "."
            Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If
    End If

    objFSO.CopyFile sourceFileName, destinationFileName, True
    strMessage = errSource & "Cannot copy " & sourceFileName & " to " & destinationFolderName & ". Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

'File was copied successfully
CopyFileToFolder = SUCCESS

End Function

Function CopyFolderToFolder(ByVal sourceFolderName, ByVal destinationFolderName)
    ' Function to copy folder with content to specified folder

    ' [sourceFolderName]       In - string, source folder name with full path
    ' [destinationFolderName]  In - string, destination folder name
    '
    ' Return values:
    '   SUCCESS - when folder is copied successfully
    '   FAIL - on error

    On Error Resume Next

    Dim errSource, strMessage, objFSO, folderName, destinationFullFolderName


    errSource = "CommonFunctions::CopyFolderToFolder "
    CopyFolderToFolder = FAIL

    'Validate arguments
    If ( IsValidString(sourceFolderName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'sourceFolderName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( IsValidString(destinationFolderName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'destinationFolderName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Check if source folder exist
    If Not objFSO.FolderExists(sourceFolderName) Then
        strMessage = errSource & "Source folder " & sourceFolderName & " does not exist. Cannot continue copy operation. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    ' Get source folder name without full path
    folderName = objFSO.GetFolder(sourceFolderName).Name

    ' Get full destination folder name
    destinationFullFolderName = objFSO.BuildPath(destinationFolderName, folderName)

    ' If folder already exists in destination location warning will be logged
    If objFSO.FolderExists(destinationFullFolderName) Then
        strMessage = errSource & "Destination folder " &  destinationFullFolderName & " exists. Files will be overwritten. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, WARNING_C)
    End If

    ' Function will copy folder content recursively
    objFSO.CopyFolder sourceFolderName, destinationFullFolderName, True

    strMessage = errSource & "Cannot copy " & sourceFolderName & " to " & destinationFullFolderName & ". Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

'Folder was copied successfully
CopyFolderToFolder = SUCCESS

End Function

Function ExportItemsToDataStore(ByRef itemName)
    ' Function to export files or folders to Datastore

    ' [itemName]  In - name(s) of file(s) or folder(s) to copy. Can be string or array

    ' Return values:
    '   SUCCESS - when all files were copied successfully
    '   FAIL - on error

    On Error Resume Next

    Dim errSource, strMessage, objFSO, appUpgradeFolder, iter, res

    ExportItemsToDataStore = SUCCESS

    errSource = "CommonFunctions::ExportItemsToDataStore "

    appUpgradeFolder = UPGRADE_DATA_STORE_FOLDER & "\" & appName

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    On Error Resume Next

    If (IsArray(itemName) = True) Then
        For iter = 0 To UBound(itemName)
            If Not IsValidString(itemName(iter)) Then
                strMessage = errSource & "At least one entry in 'itemName' array is Null or Empty." & " Application name: " & appfileName
                Call WriteToLogFile(strMessage, ERROR_C)
                Err.Clear
                Exit Function
            End If

            If objFSO.FileExists(itemName(iter)) Then
                res = CopyFileToFolder(itemName(iter), appUpgradeFolder)
            Elseif objFSO.FolderExists(itemName(iter)) Then
                res = CopyFolderToFolder(itemName(iter), appUpgradeFolder)
            Else
                Call WriteToLogFile("Array item " & itemName(iter) & " is not file or folder. Cannot continue", ERROR_C)
                res = FAIL
            End If

            If ( res <> SUCCESS ) Then
                Call WriteToLogFile("Cannot copy item: " & itemName(iter) & " to datastore", ERROR_C)
                ExportItemsToDataStore = FAIL
                Exit Function
            End If

        Next
    Else
        If Not IsValidString(itemName) Then
            strMessage = errSource & "Null or Empty value passed to parameter 'itemName'."
            Call WriteToLogFile(strMessage, ERROR_C)
            Err.Clear
            Exit Function
        End If

        If objFSO.FileExists(itemName) Then
            res = CopyFileToFolder(itemName, appUpgradeFolder)
        Elseif objFSO.FolderExists(itemName) Then
            res= CopyFolderToFolder(itemName, appUpgradeFolder)
        Else
            Call WriteToLogFile("Item " & itemName & " is not file or folder. Cannot continue", ERROR_C)
            res = FAIL
        End If

        If ( res <> SUCCESS ) Then
            Call WriteToLogFile("Cannot copy file " & itemName & " to folder", ERROR_C)
            ExportItemsToDataStore = FAIL
            Exit Function
        End If
    End If

End Function

Function ExportRegKey(ByVal keyName, ByVal fileName)
    ' Function to export registry key to file

    ' [keyName]   In - registry key to copy
    ' [fileName]  In - file to which registry key will be exported

    ' Return values:
    '   SUCCESS - when registry key was exported successfully
    '   FAIL - on error

    On Error Resume Next

    Dim WshShell, cmdExportRegistry, strMessage, errSource, ret

    errSource = "CommonFunctions::ExportRegKey "
    ExportRegKey = FAIL

    'Validate arguments
    If ( IsValidString(keyName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'keyName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( IsValidString(fileName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'fileName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ("XP" = GetOSVersion() ) Then
        cmdExportRegistry = "reg.exe EXPORT " & """" & keyName & """" & " " & """" & fileName & """"
    Else
    cmdExportRegistry = "reg.exe EXPORT " & """" & keyName & """" & " " & """" & fileName & """" & " /y"
    End If

    ' Check for errors
    strMessage = errSource & "Could not check operating system version. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Set WshShell = CreateObject("Wscript.Shell")

    ret = WshShell.Run(cmdExportRegistry, 0, True)

    strMessage = errSource & "Cannot export " & keyName & " to " & fileName & ". Application name: " & appName & "."
    If ( ret <> 0 ) Then
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    ExportRegKey = SUCCESS

End Function

Function ExportRegKeyToDatastore(ByVal keyName, ByVal fileName)
    ' Function to export registry key to data store

    ' [keyName]   In - registry key to copy
    ' [fileName]  In - file in Datastore to which registry key will be exported

    ' Return values:
    '   SUCCESS - when registry key was exported successfully
    '   FAIL - on error

    On Error Resume Next

    Dim strMessage, errSource, ret, fullFileName

    errSource = "CommonFunctions::ExportRegKeyToDatastore "
    ExportRegKeyToDatastore = FAIL

    'Validate arguments
    If ( IsValidString(keyName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'keyName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( IsValidString(fileName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'fileName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    fullFileName = UPGRADE_DATA_STORE_FOLDER & "\" & appName & "\" & fileName

    If(ExportRegKey(keyName, fullFileName) <> SUCCESS) Then
        strMessage = errSource & "Cannot export " & keyName & " to " & fullFileName & ". Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    ExportRegKeyToDatastore = SUCCESS

End Function

Function ImportRegKey(ByVal fileName)
    ' Function to import registry key from file

    ' [fileName]    In - file name with exported registry key(s)

    ' Return values:
    '   SUCCESS - registry was imported successfully
    '   FAIL - on error

    On Error Resume Next

    Dim WshShell, strMessage, errSource, cmdImportRegistry, ret, objFSO, appBitInfo

    errSource = "CommonFunctions::ImportRegKey "
    ImportRegKey = FAIL

    'Validate arguments
    If ( IsValidString(fileName) = False ) Then
        Call WriteToLogFile(errSource & "argument 'fileName' is not valid string", ERROR_C)
        Err.Clear
        Exit Function
    End If

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Check if source file exists
    If Not objFSO.FileExists(fileName) Then
        strMessage = errSource & "File " & sourceFileName & " does not exist. Cannot continue registry key(s) import operation. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    appBitInfo = GetAppBitInfo(appName)

    If ("XP" = GetOSVersion() ) Then
    cmdImportRegistry = "reg.exe IMPORT " & """" & fileName & """"
    Else
        cmdImportRegistry = "reg.exe IMPORT " & """" & fileName & """" & " /reg:" &  appBitInfo
    End If

    ' Check for errors
    strMessage = errSource & "Could not check operating system version. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Set WshShell = CreateObject("Wscript.Shell")

    strMessage = errSource & "Failed to create Wscript.Shell object. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If(WshShell.Run(cmdImportRegistry, 0, True) <> 0) Then
        strMessage = errSource & "Cannot import registry key(s) from file: " & fileName & " Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    ImportRegKey = SUCCESS

End Function

Function UpgradeDataStoreCleanup()
    ' Function to cleanup upgrade datastore folder

    ' Return values:
    '   SUCCESS - cleanup finished successfully
    '   FAIL - on error

    On Error Resume Next

    Dim objFSO, strMessage, errSource, appDataStoreDir

    errSource = "CommonFunctions::UpgradeDataStoreCleanup "
    UpgradeDataStoreCleanup = FAIL

    appDataStoreDir = UPGRADE_DATA_STORE_FOLDER & "\" & appName

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    strMessage = errSource & "Failed to create Scripting.FileSystemObject object. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If Not (objFSO.FolderExists(appDataStoreDir)) Then
        UpgradeDataStoreCleanup = SUCCESS
        Exit Function
    End If

    strMessage = errSource & "Failed to check if folder " & appDataStoreDir & " exists. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    objFSO.DeleteFolder appDataStoreDir, True
    If (Err.Number <> 0) Then
        Call WriteToLogFile("Unable to delete application data store folder. Error number is " & Err.Number & " : " & Err.Description, WARNING_C)
        Err.Clear
        Exit Function
    End If

UpgradeDataStoreCleanup = SUCCESS

End Function

Function IsMajorOsUpgrade()
    ' Function to check if it is major OS upgrade

    ' Return values:
    '   TRUE  - data retention parition exists and it is major OS upgrade scenario
    '   FALSE - data retention partition doesn't exist or data retention partition exists
    '           and it is not major OS upgrade scenario
    '   Null  - error occured
    On Error Resume Next

    Dim objFSO, errSource, strMessage, appDataFile, currentOSVersion, previousOSVersion, metaData, metaDataFile, ret, sysVersionCompRes

    errSource = "CommonFunctions::IsMajorOsUpgrade "
    IsMajorOsUpgrade = Null

    metaDataFile = UPGRADE_DATA_STORE_FOLDER & "\" & appName & "\" & UPGRADE_DATA_STORE_FILE

    ret = CheckIfDriveExists(UPGRADE_DATA_STORE_DRIVE, DRIVE_FIXED)

    ' check for errors
    If (IsNull(ret)) Then
        strMessage = "Cannot check if drive " & UPGRADE_DATA_STORE_DRIVE & " exist. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Err.Clear
        Exit Function
    End If

    If ( ret = False ) Then
        ' check for errors
        strMessage = "Drive " & UPGRADE_DATA_STORE_DRIVE & " does not exist. This it not Major OS Upgrade scenario. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, INFO_C)
        Err.Clear
        IsMajorOsUpgrade = FALSE
        Exit Function
    End If

    ' Create FileSystemObject
    Set objFSO = CreateObject("Scripting.FileSystemObject")
    If (CheckForErrors("Failed to create Scripting.FileSystemObject") = True) Then
        Exit Function
    End If

    If objFSO.FileExists(metaDataFile) Then

        ' Get properties (meta date) from meta file
        Set metaData = GetPropertiesFromFile(metaDataFile, UPGRADE_DATA_STORE_SEPARATOR)

        ' Get current OS version
        currentOSVersion = GetOSVersionNumber()

        strMessage = errSource & "Could not check operating system version. Application name: " & appName & "."
        If (CheckForErrors(strMessage) = True) Then
            Exit Function
        End If

        ' Get previous OS version
        previousOSVersion = metaData("OSVersionNumber")

        ' Compare OS version to determine major OS upgrade scenario

        sysVersionCompRes = SysVersionComp(previousOSVersion, currentOSVersion)

        If (IsNull(sysVersionCompRes)) Then
            strMessage = errSource & "OS version comparison failed. Application name: " & appName & "."
            Call WriteToLogFile(strMessage, ERROR_C)
            Exit Function
        End If

        If (sysVersionCompRes = -1) Then
        ' previousOSVersion < currentOSVersion - it is a major OS upgrade scenario
            IsMajorOsUpgrade = TRUE
        Else
            IsMajorOsUpgrade = FALSE
        End If
    Else
            IsMajorOsUpgrade = FALSE
    End If

End Function

Function SysVersionComp(ByVal sysVer1, ByVal sysVer2)
    ' Compares two system versions and returns a value that represents the result of the comparison.
    '
    ' [sysVer1] In - system version 1
    ' [sysVer2] In - system version 2
    '
    ' Return values:
    '  -1 (if sysVer1 < sysVer2)
    '  0 (if sysVer1 = sysVer2)
    '  1 (if sysVer1 > sysVer2)
    '  Null (if sysVer1 or sysVer2 is Null or sysVer1 or sysVer2 has invalid format).
    '
    ' Remarks:
    '  - Return values are based on build in StrComp function.
    '  - http://www.w3schools.com/vbscript/func_strcomp.asp

    Dim versionPattern, errSource, ret, strMessage, splittedSysVer1, splittedSysVer2, version

    On Error Resume Next

    errSource = "CommonFunctions::SysVersionComp: "
    SysVersionComp = Null

    versionPattern = "^\d{1,3}\.\d{1,3}\.\d{1,5}?$"

    ret = InStrRegex(sysVer1, versionPattern)

    ' check for errors
    strMessage = errSource & "Could not check if sysVer1 is valid system version."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Return false if versionPattern was not found in sysVer1 at position 0
    If (ret <> 0) Then
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    ret = InStrRegex(sysVer2, versionPattern)

    ' check for errors
    strMessage = errSource & "Could not check if sysVer2 is valid system version."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Return false if versionPattern was not found in sysVer at position 0
    If (ret <> 0) Then
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    splittedSysVer1 = Split(sysVer1, ".")
    splittedSysVer2 = Split(sysVer2, ".")


    ' Convert splittedSysVer elements to Long
    splittedSysVer1(0) = CLng(splittedSysVer1(0))
    splittedSysVer1(1) = CLng(splittedSysVer1(1))

    splittedSysVer2(0) = CLng(splittedSysVer2(0))
    splittedSysVer2(1) = CLng(splittedSysVer2(1))

    ' assume that versions are equal
    SysVersionComp = 0

    ' Two components of OS version number are compared
    For version = 0 To 1
        If (splittedSysVer1(version) > splittedSysVer2(version)) Then
            ' set return value
            SysVersionComp = 1
            Exit Function
        ElseIf (splittedSysVer1(version) < splittedSysVer2(version))Then
            ' set return value
            SysVersionComp = -1
            Exit Function
        End If
    Next

End Function

Function ExistsRegValue(ByVal keyName, ByVal value)
    ' Function to check whether valu exists in registry key

    ' [keyName] In - registry key to check
    ' [value]   In - value to search for in [keyName]

    ' Return values:
    '   Null  - Failed to check
    '   True  - exists
    '   False - does not exist

    On Error Resume Next

    Dim oReg, strMessage, errSource, ret, strValue, WshShell, cmd, keyNameEscaped, valueEscaped


    errSource = "CommonFunctions::ExistsRegValue "

    ExistsRegValue = Null

    'Validate function parameters
    If Not IsValidString(keyName) Then
        strMessage = errSource & "Null or Empty value passed to parameter 'keyName'."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    'Validate function parameters
    If Not IsValidString(value) Then
        strMessage = errSource & "Null or Empty value passed to parameter 'value'."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    Set WshShell = CreateObject("Wscript.Shell")

    strMessage = errSource & "Failed to get shell object. Application name: " & appName & "."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    'escape '"' characters for keyName and Value
    keyNameEscaped = Replace(keyName,Chr(34),Chr(92) & Chr(34))
    ' Check for errors
    strMessage = errSource & "Replace function failed for string keyName. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    valueEscaped = Replace(value,Chr(34),Chr(92) & Chr(34))
    ' Check for errors
    strMessage = errSource & "Replace function failed for string value. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    cmd = "cmd /c reg query " & Chr(34) & keyNameEscaped & Chr(34) & " /v " & valueEscaped
    ret = WshShell.Run(cmd, 0, true)

    ' Check for errors
    strMessage = errSource & "Failed to call command: " & cmd &". Application name: " & appName & "."
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If (ret <> 0) Then
        ExistsRegValue = False
    Else
        ExistsRegValue = True
    End If
End Function

Function ExportRegValue(ByVal keyName, byVal value, ByVal fileName)
    ' Function to export registry value to file

    ' [keyName]  In - registry key to check
    ' [value]    In - value in [keyName] to export
    ' [fileName] In - full path to target file for export

    ' Return values:
    '   SUCCESS - export succeeded
    '   FAIL    - export failed

    On Error Resume Next

    Dim wshShell, cmdExportRegistry, strMessage, errSource, returnStatus, iter, objFSO, ret, fileProperties, fileHandle

    errSource = "CommonFunctions::ExportRegValue "

    ExportRegValue = FAIL

    'Validate function parameters
    If Not IsValidString(keyName) Then
        strMessage = errSource & "Null or Empty value passed to parameter 'keyName'."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    If Not IsValidString(value) Then
       strMessage = errSource & "Null or Empty value passed to parameter 'value'." & " Application name: " & appName
       Call WriteToLogFile(strMessage, ERROR_C)
       Exit Function
    End If

    'Export the whole key. Then not necessary values from resulting *.reg file will be erased.
    If (SUCCESS <> ExportRegKey(keyName, fileName)) Then
        Exit Function
    End If

    'Reg file is composed in the following manner:
    '  line l  |   Windows Registry Editor Version 5.00
    '  line 2  |   [keyName]
    '  line 3  |   "value_1"="xxx"
    '  line 4  |   "value_2"="yyy"
    '  line ...|     ...
    '  line n  |   "value_n"="zzz"
    'It is necessary to remove everything except lines 1-3

    Set objFSO = CreateObject("Scripting.FileSystemObject")

    ' Check for errors
    strMessage = errSource & "Failed to create Scripting.FileSystemObject. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    If Not (objFSO.FileExists(fileName)) Then
        strMessage = errSource & fileName & "does not exist after export. Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    ' Check for errors
    strMessage = errSource & "Failed to check if " & fileName & "exists after export. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' Ensure that deployment.properties file is "writable"
    Set fileNameProperties = objFSO.GetFile(fileName)

    If (SUCCESS <> RemoveFileAttribute(fileName, Array(READ_ONLY, HIDDEN))) Then
        Call WriteToLogFile("Failed to change file " & fileName & " attribute(s)", ERROR_C)
        Exit Function
    End If

    Set fileHandle = objFSO.OpenTextFile(fileName, FOR_READING, False, TRISTATE_TRUE)

    ' Check for errors
    strMessage = errSource & "Failed to open " & fileName & " for reading. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    ' store in linesToRetain only lines, we want
    Dim linesToRetain, strLine, keyNamefound, valueEscaped
    linesToRetain = ""
    keyNamefound = False
    valueEscaped = Replace(value,Chr(34),Chr(92) & Chr(34))

    ' Check for errors
    strMessage = errSource & "Replace function failed. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    Do Until fileHandle.AtEndOfStream
        strLine = fileHandle.ReadLine
        If (keyNamefound = False) Then
            linesToRetain = linesToRetain & strLine & vbCrLf
            If (1 = InStr(1, strLine, "[")) Then
                keyNamefound = True
            End If
        Else
            If (1 = InStr(1, strLine, Chr(34) & valueEscaped & Chr(34))) Then
                linesToRetain = linesToRetain & strLine & vbCrLf
            End If
        End If
    Loop
    fileHandle.Close

    ' store linesToRetain in fileName
    Set fileHandle = objFSO.OpenTextFile(fileName, FOR_WRITING, True)

    ' Check for errors
    strMessage = errSource & "Failed to open " & fileName & " for writing. Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    fileHandle.Write(linesToRetain)

    ' Check for errors
    strMessage = errSource & "Failed to write content to "  & fileName & ". Application name: " & appName
    If(CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    fileHandle.Close

    ExportRegValue = SUCCESS
End Function

Function ExportRegValueToDataStore(ByVal keyName, byVal value, ByVal fileName)
    ' Function to export registry value to file, with location based on upgrade folder for application

    ' [keyName]  In - registry key to check
    ' [value]    In - value in [keyName] to export
    ' [fileName] In - relative path (depends on application name) to target file for export

    ' Return values:
    '   SUCCESS - export succeeded
    '   FAIL    - export failed

    On Error Resume Next

    Dim strMessage, errSource, fullFileName

    ExportRegValueToDataStore = FAIL

    errSource = "CommonFunctions::ExportRegValueToDataStore "

    fullFileName = UPGRADE_DATA_STORE_FOLDER & "\" & appName & "\" & fileName

    If (ExportRegValue(keyName, value, fullFileName) <> SUCCESS) Then
        strMessage = errSource & "Cannot export registry value " & value & " for key " & keyName & " to the file " & fullFileName & ". Application name: " & appName & "."
        Call WriteToLogFile(strMessage, ERROR_C)
        Exit Function
    End If

    ExportRegValueToDataStore = SUCCESS
End Function

Function addRegistryEntries(registryEntries)
    'Function adds specific keys to the registry. Error check.
    '  Handled VB data types:
    '       Long, Integer - put as REG_DWORD (32-bit number)
    '       Other - put as string - REG_SZ (this is default)
    '
    'Parameters:
    '  [registryEntries] In - dictionary which consists of: Registry key path - Key value.
    '

    On Error Resume Next

    Const REG_VALUE_TYPE_DEFAULT = "REG_SZ"
    addRegistryEntries = FAIL  ' Think positive ;P

    Dim wshShell, key, regKeys, keyType, strMessage
    Set wshShell = CreateObject( "WScript.Shell" )
    regKeys = registryEntries.Keys

    'Check for Errors
    strMessage = "Could not find proper registry keys."
    If (CheckForErrors(strMessage) = True) Then
        Exit Function
    End If

    'Add registry keys
    For Each key In regKeys
        Dim keyValue, valueType
        valueType = REG_VALUE_TYPE_DEFAULT
        keyValue = registryEntries(key)

        'Check key value type and for data type "Long"
        If (TypeName(keyValue) = "Long" Or TypeName(keyValue) = "Integer") Then
            valueType = "REG_DWORD"
        End If

        wshShell.RegWrite key, keyValue, valueType

        'Check for Errors
        strMessage = "Could not add value "& keyValue & " of "& valueType & " to "& key
        If (CheckForErrors(strMessage) = True) Then
            Exit Function
        End If

        strMessage = "Value type "& valueType &" added for "& key
        Call WriteToLogFile(strMessage, INFO_C)
    Next

    addRegistryEntries = SUCCESS

End Function
